
#-------------------------------------------
# (c) Peter Campbell Software 2001-2004, all rights reserved.
# http://www.fastbase.co.nz
#-------------------------------------------


#====================================================================#
# procedures to create windows, input fields: char, number, lookup etc.
#====================================================================#

# Initialise window/gui setup variables, fonts and tk options
proc initialise_widgets { } {
	global widgets_initialised
	global setup

	if {[info exists widgets_initialised]} { return }
	set widgets_initialised 1

	# Delay drawing of windows until input_window is called; also defaulted in event.tcl
	setup delayed_draw $setup(fbproxy)

	# Default colours
	setup select_colour "lightsteelblue"		;# The colour of the highlight bar on selection screens
	setup alert_colour "lightgoldenrod"			;# The colour of alert messages (eg: customer alerts, product alerts etc)
	setup alert_edit "lightgoldenrodYellow"		;# The colour of the alert window when mouse over for editing
	setup output_bg "lightyellow"				;# The background colour of the output windows (reports and stationery)

	# Default font used on all screens, usually set in setup.dat and varies depending on screen resolution
	setup font {Arial 12}
	setup text_font {"Courier New" 9}

	# Create bold font
	# Shouldn't need catch ...
	catch { option add *font $setup(font) startupFile }
	set setup(font_bold) [linsert $setup(font) end bold]

	# Set disabled foreground & background colour (required when using tcl >= 8.4)
	# catch errors (ie: running from tclsh rather than wish)
	# Shouldn't need catch ...
	catch {
		option add *DisabledForeground black
		option add *DisabledBackground lightgray
	}

	# Create small font (1 or 2 sizes smaller)
	# Also in printing init function ?
	set size [lindex $setup(font) 1]
	set size [expr {$size >= 10} ? {9} : {$size - 1}]
	set setup(font_small) [lreplace $setup(font) 1 1 $size]
	set setup(font_small_bold) [lreplace $setup(font) 1 1 $size bold]
}

proc NULL { args } {}

# this procedure is used to create a new window
# wname = name of an array variable for storing the window information
# title = window title
# args	= options for the window command - to be defined...

proc create_window { wname pname title args } {
	global setup unique system

	# get the calling namespace and set the window in the global system array
	initialise_widgets
	set ns [uplevel {namespace current}]

	if { [array names system $ns,$pname] != {} } {
		if { !$::event::limbo } { event::activate_trapper $system($ns,$pname) }
	}

	# check that the window name doesn't already exist - do this to trap unwanted windows being opened
	if {![catch { set system($ns,$wname) } ]} {
		error "The system is attempting to create a window with a name that already exists, reference \"$ns\::$wname\"."
	}

	# new window name (will be either a frame or toplevel)
	set w .window[incr unique]

	# get & set the window options
	system_set_defaults $w bg background {popup 0} position [list escape $wname] {draw_now 0}
	foreach {option value} [system_set_options $w $args] {
		error "Invalid option: $option, value = $value"
	}

	# store the window toplevel in the system array
	set system($ns,$wname) $w

	# some additional options that require setting
	set system($w,ns) $ns
	set system($w,unique) 0
	set system($w,redraw_command) ""
	set system($w,focus) ""
	set system($w,pname) $pname
	set system($w,centre) 0
	set system($w,drawn) 0
	set system($w,parent) $w

	if { !$setup(delayed_draw) } {
		set system($w,draw_now) 1
	}

	set system($w,draw_commands) {}
	set command [list draw_window $w $title $wname]
	if { $system($w,draw_now) } {
		eval $command
	} else {
		lappend system($w,draw_commands) [list $command $w]
	}

	# draw dummy 'window' to trap user-generated events
	# create_event_trapper $w

	return $w
}

proc draw_window {w title wname} {
	global setup system
	set ns $system($w,ns)

	# record the original title for instrumentation
	set original_title $title

	# create the new toplevel (set and hide/withdraw it for now) or frame
	# only -position & -popup windows create new toplevels
	if {$system($w,popup) || $system($w,position) != ""} {
		set toplevel 1
		set destroy "destroy_window $w $toplevel"

		toplevel $w
		wm title $w $title
		wm transient $w .

		# for transient windows we need to do a "grab" to stop the window behind getting any events
		set current_grab [grab current]

		# have to wait till window is visible on linux operating systems
		while { [catch { grab $w }] } {
			logToFile "grab error"
			tkwait visibility $w
		}

		if {$current_grab != ""} {
			::event::bind $w <Destroy> "+grab_original $w %W [lindex $current_grab 0]"
		}

		# if the current window is also a transient then add a focus command to the destroy script
		set current [lindex $setup(frame_stack) end]
		set currentf [lindex $current 0]
		if {$currentf != ""} {

			# get the focus of the current window
			# add a destroy command to reset the focus to the same field when it is re-displayed
			if {[catch { set current_focus [focus -displayof $currentf] }]} {
				set current_focus ""
			}
			append destroy ";[list focus_force $current_focus]"
		}

		# create the main bindings for the control-function keys (enquiries)
		# note: we catch errors, this is for special exception when running remote update (see fbclient)
		# because we won't be logged into a company yet, ie: no as_module() records
		catch {login::create_toplevel_bindings $w}
	} else {
		set toplevel 0
		set destroy ""

		frame $w

		# append company name to the window title
		if {$wname != "MAIN"} {
			if {[array names login::company NAME] != ""} {
				append title " - $login::company(NAME)"
			}
		}

		# Display the logged-in user's id in the window title
		if { [setup window_user_display 0] && [info exists login::user_name] && $login::user_name != ""} {
			set title "$login::user_name: $title"
		}
		
		wm title . $title

		# pack forget the current open window
		while {1} {
			set current [lindex $setup(frame_stack) end]
			set currentf [lindex $current 0]

			if {$currentf == ""} { break }
			if { ![winfo exists $currentf] } {
				# remove last invalid entry from the stack
				set setup(frame_stack) [lreplace $setup(frame_stack) end end]
				continue
			}

			# if the current window is a toplevel then withdraw it (simulate a "close")
			if {[winfo toplevel $currentf] == $currentf} {
				wm withdraw $currentf
				set setup(frame_stack) [lreplace $setup(frame_stack) end end]
				grab .

				set destroy "[list reshow_transient $currentf $current];$destroy"
				continue
			}

			# not a top-level
			break
		}

		if {$currentf != ""} {
			# get the focus of the current window
			# add a destroy command to reset the focus to the same field when it is re-displayed
			if {[catch { set current_focus [focus -displayof $currentf] }]} {
				set current_focus ""
			}
			set destroy "destroy_window $w $toplevel;$destroy;[list focus_force $current_focus]"

			pack forget $currentf

			# remove the menu from the window
			. configure -menu ""
		}

		pack $w -expand yes -fill both
	}

	# add the "window" to the frame stack
	lappend setup(frame_stack) [list $w $title $original_title]

	::event::bind $w <Destroy> +$destroy

	# -bg or -background option?
	if {$system($w,bg) != ""} { set system($w,background) $system($w,bg) }
	if {$system($w,background) != ""} {
		$w configure -bg $system($w,background)
		option add *[string trim $w .].*Background $system($w,background)
	}

	# -position?
	if {$system($w,position) != ""} {
		# position window as specified, "+x+y" notation (see "wm geometry" command)
		wm geometry $w $system($w,position)
	}

	# -popup?
	if {$system($w,popup)} {
		# withdraw "popup" window
		wm withdraw $w
		# specify that the window is to be centered later (see input_window)
		set system($w,centre) 1
	}

	# create binding so that after any event we "update_window"
	set toplevel [winfo toplevel $w]
	::event::bind $toplevel <KeyRelease> "update_window %W"
	::event::bind $toplevel <ButtonRelease> "update_window %W"

	::event::bind $w <Destroy> "+array unset system %W*;array unset system_calc %W*;array unset system_noentry %W*;array unset system_bindings %W*"

	# remove window from the system array
	::event::bind $w <Destroy> "+destroy_window_event %W $w $ns $wname"

	# -escape "window"?
	if {$system($w,escape) != ""} {
		new_bind $w <Escape> "namespace eval $ns \{ close_window $system($w,escape) destroy \}"
	}

	# trap events
	event::create_trapper $w

	set system($w,drawn) 1
}

proc focus_force { w } {
	catch { focus -force $w }
}

proc destroy_window_event { W w ns wname } {
	global system

	# only unset the system array if the destroy event is for the parent window
	if {$W == $w} {
		array unset system $ns,$wname
	}
}

proc grab_original { w1 w2 grabw } {
	if {$w1 != $w2} { return }
	catch { grab $grabw }
}

proc reshow_transient { w current } {
	global setup

	lappend setup(frame_stack) $current

	# re-display the window and set the grab
	# check that the window exists first, as we had some errors occuring at Briteuro, not sure why yet
	# see also: login/close_xxx
	if {[winfo exists $w]} {
		wm deiconify $w
		grab $w
	}
}

proc destroy_window { w toplevel } {
	global setup

	set data [lindex $setup(frame_stack) end]
	set win [lindex $data 0]

	# check the window being destroyed is the current top "window"
	if {$win != $w} {
		# is this an error or is the destroy event getting called multiple times?
		return
	}

	# remove the last entry from the frame stack
	set setup(frame_stack) [lreplace $setup(frame_stack) end end]

	if {$toplevel} { return }

	# check if any more windows open?
	if {![llength $setup(frame_stack)]} {
		destroy .
		return
	}

	set data [lindex $setup(frame_stack) end]
	set w [lindex $data 0]
	set title [lindex $data 1]

	wm title . $title

	# check if the menu is to be re-displayed
	if {[llength $setup(frame_stack)] == 1} {
		. configure -menu .menu
	}

	# re-display the previous "window"
	# catch errors (not sure why, but Carey from corpcons has got errors here, to do with zeacom/popups)
	catch {
		pack $w -expand yes -fill both
	}
}

proc new_bind { window event command } {
	global system_bindings

	set toplevel [winfo toplevel $window]
	set topw [top_window $window]

	# determine if this command is already specified for this window/binding, if it is then don't add it again
	# an example of a duplicate binding occurs when changing company (the F1,F2... bindings for enquiries)
	set add_command 1
	if {![catch { set commands $system_bindings($window,$event) }]} {
		# get list of commands
		if {[lsearch $commands $command] != -1} {
			set add_command 0
		}
	}

	if {$add_command} {
		lappend system_bindings($window,$event) $command
	}

	# get the current list of bindings for this window/event, add "new_bind_event" if not there already
	set bindings [bind $toplevel $event]
	if {![string match "*new_bind_event*" $bindings]} {
		::event::bind $toplevel $event "+new_bind_event %W $event"
	}
}

proc new_bind_event { window event } {
	global setup system_bindings

	# get the current window (top screen)
	set currentf [lindex [lindex $setup(frame_stack) end] 0]

	# if the current window matches the binding then execute the command(s)
	foreach name [lsort [array names system_bindings *,$event]] {
		set w [lindex [split $name ","] 0]
		set topw [top_window $w]
		if {$topw == $currentf} {
			# detect errors, after evaluating cmd system bindings may get deleted from under us (eg: if command closes the current window)
			if {![catch { set bindings $system_bindings($name) }]} {
				foreach cmd  $bindings {
					# evaluate each command
					eval $cmd
				}
			}
		}
	}
}

#====================================================================#

proc centre_window { w } {
	if {[info commands $w] == ""} { return }

	set xmax [winfo screenwidth $w]
	set ymax [winfo screenheight $w]
	set x [expr {($xmax - [winfo reqwidth $w]) / 2}]
	set y [expr {($ymax - [winfo reqheight $w]) / 2}]

	wm geometry $w "+$x+$y"
}

#====================================================================#

# this procedure is used to create a new notebook within a window
# fName = name of an array variable for storing the frame information
# wname = name of the parent window array
# title = window title
# args	= options for the window command - to be defined

# the command results in extra arrays being created for each tab
# each array represents a new window, original = lw, tabs = lw1, lw2 etc.

proc create_notebook { wname args } {
	# get the namespace of the calling procedure, this will be used when
	# referencing all widget variables, stored in "wname(ns)"
	global system
	set ns [uplevel {namespace current}]

	# define a window name
	set f [system_new_window $wname "tab"]

	# define all options
	system_set_defaults $f tabs {width 550} {height 300}
	set unused [system_set_options $f $args]

	set pages 0
	set page_list {}

	foreach page $system($f,tabs) {
		lappend page_list $page

		set tn $f.body.f[incr pages]
		set vname $wname$pages

		set system($tn,unique) 0
		set system($ns,$vname) $tn
		system_parent_window $tn $ns $wname
	}

	set command "[list draw_notebook $f $page_list] $unused"
	system_input_widget $f $ns $command $wname

	return $f
}

proc draw_notebook { f pages args} {
	global system

	Rnotebook:create $f -tabs $pages -width $system($f,width) -height $system($f,height)
	pack $f -padx 0 -pady 0 -expand yes -fill both

	# process all options except -tabs now that widget is created
	foreach { option value } $args {
		$f configure $option $value
	}

	# add keyboard bindings for next/previous tab pages? control-left/right
	new_bind $f <Control-Left> "Rnotebook:raise $f -1"
	new_bind $f <Control-Right> "Rnotebook:raise $f +1"

	set system($f,drawn) 1
}

#====================================================================#

# this procedure is used to create a new frame within a window
# fName = name of an array variable for storing the frame information
# wname = name of the parent window array
# title = window title
# args	= options for the window command - to be defined

proc create_frame { fname wname title args } {
	global system setup
	set ns [uplevel {namespace current}]

	# define a window name
	set f [system_new_window $wname]

	# define all options
	system_set_defaults $f {pad 0} {side "top"} {anchor "nw"} background bg {expand "no"} {border 0} {fill "both"} {padx ""} {progress 0}

	# store options and process undefined ones
	set draw_args [system_set_options $f $args]

	# save the standard options required for windows/frames
	set system($f,unique) 0
	set system($f,ns) $ns
	# get the calling namespace and set the window in the global system array
	set system($ns,$fname) [list $f $f]

	set command "[list draw_frame $f $fname $title] $draw_args"
	system_input_widget $f $ns $command $wname

	return $f
}

proc draw_canvas { f fname args } {
	global system setup

	set ns $system($f,ns)
		canvas $f 
		set fc $f

	# process undefined options
	foreach {option value} $args {
		$f configure $option $value
	}

	# if -padx not set then use -pad setting
	if {$system($f,padx) == ""} { set system($f,padx) $system($f,pad) }

	pack $f -padx $system($f,padx) -pady $system($f,pad) \
		-side $system($f,side) -anchor $system($f,anchor) \
		-expand $system($f,expand) -fill $system($f,fill)


	# -bg or -background option?
	if {$system($f,bg) != ""} { set system($f,background) $system($f,bg) }
	if {$system($f,background) != ""} {
		$f configure -bg $system($f,background)
		option add *[string trim $f .].*Background $system($f,background)
	}

	# -border? this option may be used if there is no frame title to still draw a groove border
	if {$system($f,border)} {
		$f configure -relief groove -borderwidth 2
	}

	set system($f,drawn) 1
}

proc draw_frame { f fname title args } {
	global system setup

	set ns $system($f,ns)

	# either use our special labelled frame widget or the standard frame?
	if {$title != ""} {
		# test for win98, if so remove the text as it stuffs things up
		if {[setup labelframe_bug 0]} {
			global tcl_platform
			if {$tcl_platform(os) == "Windows 95"} { set title "" }
		}

		labelframe $f -text $title -font [setup font_small_bold] -relief groove -highlightthickness 2 -highlightcolor gray -padx 2
		set fc $f
	} else {
		frame $f -bd 0
		set fc $f
	}

	# process undefined options
	foreach {option value} $args {
		$f configure $option $value
	}

	# if -padx not set then use -pad setting
	if {$system($f,padx) == ""} { set system($f,padx) $system($f,pad) }

	pack $f -padx $system($f,padx) -pady $system($f,pad) \
		-side $system($f,side) -anchor $system($f,anchor) \
		-expand $system($f,expand) -fill $system($f,fill)

	# Create progress bar on frame
	if {$system($f,progress)} {
		progress::create_local $f
	}

	# -bg or -background option?
	if {$system($f,bg) != ""} { set system($f,background) $system($f,bg) }
	if {$system($f,background) != ""} {
		$f configure -bg $system($f,background)
		option add *[string trim $f .].*Background $system($f,background)
	}

	# -border? this option may be used if there is no frame title to still draw a groove border
	if {$system($f,border)} {
		$f configure -relief groove -borderwidth 2
	}

	# include binding to destroy all system variables associated with the frames components
	::event::bind $f <Destroy> "+array unset system %W*;array unset system_calc %W*;array unset system_noentry %W*;array unset system_bindings %W*"

	# remove window from the system array (same as close_window but for frames)
	::event::bind $f <Destroy> "+array unset system $ns,$fname"

	set system($f,drawn) 1
}

#====================================================================#

proc create_separator { wname } {
	global system

	set f [system_new_window $wname]
	set ns [uplevel {namespace current}]

	set command [list draw_separator $f]
	system_input_widget $f $ns $command $wname

	return $f
}

proc draw_separator { f } {
	frame $f -borderwidth 1 -relief sunken -height 2
	pack $f -padx 12 -pady 5 -fill x

	set system($f,drawn) 1
}

#====================================================================#

# wname = Fastbase name of the parent window - see "create_window", usually 'w' or similar
# field = name of the variable to store the data
# title = label text (ie: screen prompt)
# args	= options as follows:
#	-required 0/1	(force entry of field)
#	-password char	(password entry, eg: -password "*")
# 	-event {event_name command ...} generic event handling. Events are evaluated without validation

proc input_char { wname field title args } {
	global system setup

	set ns [uplevel {namespace current}]
	set f [system_new_window $wname]
	set fe $f.e

	set system($fe,ns) $ns
	set system($fe,field) $field
	set system($fe,wname) $wname

	# define all options
	system_set_defaults $fe format size width uppercase lowercase date password standard pluskey \
		validate valid {singlequote 0} required noentry enterkey enterkeycontinue enterkeytrap event range warning_range filename help help_args \
		note note_config var calc labelvar {endofmonth 0} {startofmonth 0} {email 0} {align 1} \
		labelwidth label_config {thismonth 0} {whitespace 1}

	# store options and process undefined ones
	set draw_args [system_set_options $fe $args]

	set command "[list draw_input_char $f $fe $title] $draw_args"
	system_input_widget $fe $ns $command

	return $fe
}

proc draw_input_char {f fe title args} {
	global system setup

	set ns $system($fe,ns)
	set field $system($fe,field)

	set labelw [lindex [pc_label $f $title] 1]
	entry $fe -validate all -validatecommand "entry_validation char %W %V %d %S %P"
	set_focus $fe
	pack $fe -side left -anchor nw

	foreach {option value} $args {
		$fe configure $option $value
	}

	# -password? display asterisk or whatever character was specified (value)
	if {$system($fe,password) != ""} {
		$fe configure -show $system($fe,password)
	}

	# -enterkey; process event when user tabs/enters out of field
	if {$system($fe,enterkey) != ""} {
		# note: the -enterkey option validates the screen before action
		# use: -event {Tab command} for non-validation
		set cmd "if \{\[validate_window $fe\]\} \{ $system($fe,enterkey);break \} else \{ break \}"
		::event::bind $fe <Tab> +[list namespace eval $ns $cmd]
	}

	# -enterkeycontinue; process event when user tabs/enters out of field, and don't stop unless validation fails
	if {$system($fe,enterkeycontinue) != ""} {
		# note: the -enterkeycontinue option validates the screen before action
		# use: -event {Tab command} for non-validation
		set cmd "if \{\[validate_window $fe\]\} \{ $system($fe,enterkeycontinue) \} else \{ break \}"
		::event::bind $fe <Tab> +[list namespace eval $ns $cmd]
	}

	# -enterkeytrap; trap any user events and process event when user tabs/enters out of field
	if {$system($fe,enterkeytrap) != ""} {
		# note: the -enterkeytrap option validates the screen before action
		# use: -event {Tab command} for non-validation
		set parent [top_window $fe]
		set cmd "if \{\[validate_window $fe\]\} \{ event::activate_trapper $parent; $system($fe,enterkeytrap); event::disable_trapper; break \} else \{ break \}"
		::event::bind $fe <Tab> +[list namespace eval $ns $cmd]
	}

	# -pluskey; as for -enterkey, but only enabled by setup option enable_plus_key
	if {$system($fe,pluskey) != "" && [setup enable_plus_key 0]} {
		# note: the -pluskey option validates the screen before action
		set cmd "if \{\[validate_window $fe\]\} \{ $system($fe,pluskey);break \} else \{ break \}"
		::event::bind $fe <KeyPress-plus> +[list namespace eval $ns $cmd]
	}
	
	# -event? generic event handling: -event {event_name command ...}
	if {$system($fe,event) != ""} {
		foreach {event command} $system($fe,event) {
			# execute the specified command when the event occurs
			::event::bind $fe <$event> "+if \{\[[list namespace eval $ns $command]\]\} break"
		}
	}

	# -width?
	if {$system($fe,width) != ""} {
		$fe configure -width $system($fe,width)
	}

	# -note? note to display to right of the entry box?
	if {$system($fe,note) != ""} {
		label $f.t -text $system($fe,note) -justify left -font [setup font_small]
		pack $f.t -side left
	}

	# -note_config?
	if {$system($fe,note_config) != ""} {
		eval "$f.t configure $system($fe,note_config)"
	}

	# -var?
	if {$system($fe,var) != ""} {
		label $f.v -textvariable $ns\::$system($fe,var) -justify left -font [setup font_small]
		pack $f.v -side left
	}

	# -labelvar?
	if {$system($fe,labelvar) != ""} {
		$labelw configure -textvariable $ns\::$system($fe,labelvar)
	}

	# -label_config?
	if {$system($fe,label_config) != ""} {
		eval "$labelw configure $system($fe,label_config)"
	}

	# -align 0? this option removes the widget from the packing/alignment list (see proc pack_widgets)
	if {!$system($fe,align)} {
		array unset system $labelw,pack
	}
	# -labelwidth? set the label width to the specified size, used in conjunction with "-align 0"
	if {$system($fe,labelwidth) != ""} {
		$labelw configure -width $system($fe,labelwidth)
	}

	# -date field?
	if {$system($fe,date) == "1"} {
		# display the date field (and set trace if variable changes)
		if {[info exists $ns\::$field] == 0} { set $ns\::$field "" }

		# ensure the date field is no more than 10 characters, ie: doesn't include the time 00:00:00
		set $ns\::$field [string range [set $ns\::$field] 0 9]

		$fe insert end [date::display [set $ns\::$field]]

		# entry_validation sets the variable value on key changes
		# entry_validation sets the display on focusout events
		$fe configure -validatecommand "entry_validation date %W %V %d %S %P"

		# standard validation logic checks the date validity
		set oldDate [set $ns\::$field]
		set cmd "\[date::validate \[$fe get\] $system($fe,endofmonth) $system($fe,startofmonth) $oldDate $system($fe,thismonth)\]"
		set system($fe,valiDATE) [list $cmd "Invalid Date!"]

		# set write trace to update variable on screen if changed
		set cmd "NULL $fe;update_date $ns\::$field $fe"
		trace variable $ns\::$field w $cmd
		::event::bind $fe <Destroy> +[list trace vdelete $ns\::$field w $cmd]

		# keypresses to alter the date (page up/down)
		::event::bind $fe <Key-Prior> "user_change_date $fe Prior"
		::event::bind $fe <Key-Next> "user_change_date $fe Next"

		# double click or press F1 on date field? select date from calendar
		::event::bind $fe <Control-F1> "+NULL"
		::event::bind $fe <KeyPress-F1> "+date::calendar_select $fe;break"
		::event::bind $fe <Double-1> "+date::calendar_select $fe;break"
	}

	# -time field?
	if {$system($fe,date) == "2"} {
		# display the time field (and set trace if variable changes)
		if {[info exists $ns\::$field] == 0} { set $ns\::$field "" }

		$fe insert end [set $ns\::$field]

		# entry_validation sets the variable value on key changes
		# entry_validation sets the display on focusout events
		$fe configure -validatecommand "entry_validation time %W %V %d %S %P"

		# standard validation logic checks the time validity
		set cmd "\[date::validate_time \[$fe get\]\]"
		set system($fe,valiDATE) [list $cmd "Invalid Time!"]

		# set write trace to update variable on screen if changed
		set cmd "NULL $fe;update_time $ns\::$field $fe"
		trace variable $ns\::$field w $cmd
		::event::bind $fe <Destroy> +[list trace vdelete $ns\::$field w $cmd]

		# keypresses to alter the date (page up/down)
		::event::bind $fe <Key-Prior> "user_change_time $fe Prior"
		::event::bind $fe <Key-Next> "user_change_time $fe Next"
	}

	# -filename {open/save [options]}
	if {$system($fe,filename) != ""} {
		# parameter = open or save
		# eg: -filename {open -filetypes {{SQL .sql}}}
		# F1 on entry or click on label to activate file dialog
		set value $system($fe,filename)

		switch -- [lindex $value 0] {
			"save" {
				set cmd "+set x \[tk_getSaveFile -initialdir [pwd] -initialfile \[set $ns\::$field\]"
			}
			"dir" {
				set cmd "+set x \[tk_chooseDirectory -initialdir \[set $ns\::$field\]"
			}
			default {
				set cmd "+set x \[tk_getOpenFile -initialdir [pwd] -initialfile \[set $ns\::$field\]"
			}
		}

		set options [lrange $value 1 end]

		set fb $f.b
		label $fb -text "  \[Browse ...\]  " -relief flat -bd 2 -fg darkred
		::event::bind $fb <Enter> "+$fb configure -relief raised"
		::event::bind $fb <Leave> "+$fb configure -relief flat"
		pack $fb -side left -padx 8

		# set-up bindings to activate "file dialog" window
		append cmd " $options"
		append cmd " -parent [winfo toplevel $fe]\];"
		append cmd "if \{\"\$x\" != \"\"\} \{set $ns\::$field \$x\}"
		::event::bind $fb <1> $cmd
		::event::bind $fe <Control-F1> "+NULL"
		::event::bind $fe <Key-F1> $cmd
	}

	# -help text?
	if {$system($fe,help) != ""} {
		set cmd [list popup_help $system($fe,help) $fe]
		# append all help arguments onto the command; help_args might be used for "-time 0" or other, see support.tcl - popup_help
		foreach arg $system($fe,help_args) { lappend cmd $arg }
		::event::bind $fe <FocusIn> $cmd
	}

	# -email?
	if {$system($fe,email)} {
		# provisional option to allow user to click on email and start an email message via "mailto"
		# bind $fe <Enter> "popup_help \"Click to Start an Email Message.\" $fe"
		# bind $fe <1> "exec [lindex [auto_execok start] 0] mailto:\[set $ns\::$field\] &"

		# validate the email address
		set cmd "\[validate_email \[$fe get\]\]"
		set system($fe,valiDATE) [list $cmd "Invalid Email Address!"]
	}

	# link the entryfield to the variable/field (date fields are special exceptions)
	if {$system($fe,date) == ""} {
		$fe configure -textvariable $ns\::$field
	}

	# set the size/width of the field
	# if uppercase then increase the width to allow for larger characters
	if {$system($fe,size) != "" && $system($fe,width) == ""} {
		set sizex [expr {$system($fe,size) + 2}]
		if {$system($fe,uppercase) == "1"} {
			set sizex [expr {int([expr {$sizex * 1.25}])}]
		}
		$fe configure -width $sizex
	}

	# -standard "code" text selection option?
	if {$system($fe,standard) != ""} {
		# press F1 to select from list of options
		# press F2 to maintain list of options (requires level 1 UT security)
		::event::bind $fe <Control-F1> "+NULL"
		::event::bind $fe <KeyPress-F1> "+standard_text_selection $fe"
		set f2 ""
		if {$login::user(UT_SECURITY)} {
			::event::bind $fe <Control-F2> "+NULL"
			::event::bind $fe <KeyPress-F2> "+standard_text_selection $fe maint"
			set f2 "\nPress F2 to maintain messages."
		}
		::event::bind $fe <FocusIn> "popup_help \"Press F1 to select message.$f2\" $fe -xoffset 200"
	}

	# -uppercase/lowercase?
	if {$system($fe,uppercase) == "1" || $system($fe,lowercase) == "1"} {
		# set write trace to update variable (uppercase/lowercase)
		set cmd "NULL $fe;update_case [list $ns\::$field $system($fe,uppercase)]"
		trace variable $ns\::$field w $cmd
		::event::bind $fe <Destroy> +[list trace vdelete $ns\::$field w $cmd]
	}

	# if the field isn't explicitly upper or lower case then provide a
	# pop-up menu accessible via the right-mouse-click
	set case_override [expr {$system($fe,uppercase) != "" || $system($fe,lowercase) != ""}]
	set extras [expr {$case_override} ? {""} : {"U L"}]
	::event::bind $fe <ButtonPress-3> "application_popup_entry $fe %X %Y $extras"

	set_system $fe calc
	set_system $fe noentry

	set system($fe,drawn) 1
}

proc update_date { field window name1 name2 op } {
	if {[info commands $window] != ""} {
		# save the state of the date field (-state, -validate)
		set state [$window cget -state]
		set valid [$window cget -validate]

		# make the field normal (so we can change its value) and disable validation
		$window configure -state normal -validate none
		$window delete 0 end
		$window insert end [date::display [set $field]]

		# reset the status and validation
		$window configure -state $state -validate $valid
	}
}

proc update_time { field window name1 name2 op } {
	if {[info commands $window] != ""} {
		# save the state of the time field (-state, -validate)
		set state [$window cget -state]
		set valid [$window cget -validate]

		# make the field normal (so we can change its value) and disable validation
		$window configure -state normal -validate none
		$window delete 0 end
		$window insert end [set $field]

		# reset the status and validation
		$window configure -state $state -validate $valid
	}
}

proc user_change_date { fe method } {
	global system

	set ns $system($fe,ns)
	set field $system($fe,field)

	set date [set $ns\::$field]

	if {$date == ""} { return 1 }

	switch -- $method {
		"Prior" {
			set new [date::calc $date - months 1]
			if {$date == [date::endofmonth $date]} { set new [date::endofmonth $new] }
		}
		"Next" {
			set new [date::calc $date + months 1]
			if {$date == [date::endofmonth $date]} { set new [date::endofmonth $new] }
		}
	}

	# set new date
	set $ns\::$field $new
	return 1
}

proc user_change_time { fe method } {
	global system

	set ns $system($fe,ns)
	set field $system($fe,field)

	set time [set $ns\::$field]

	if {$time == ""} { return 1 }

	# get hour
	set hour [string trimleft [string range $time 0 1] "0"]
	if {$hour == ""} { set hour 0 }

	switch -- $method {
		"Prior" {
			incr hour -1
			if {$hour < 0} { set hour 23 }
		}
		"Next" {
			incr hour
			if {$hour > 23} { set hour 0 }
		}
	}

	set new [format %02d $hour]
	append new [string range $time 2 4]

	# set new date
	set $ns\::$field $new
	return 1
}

proc update_case { field upper name1 name2 op } {
	# convert the field to upper or lower case as specified
	set case [expr {$upper == "1"} ? {"toupper"} : {"tolower"}]
	set $field [string $case [set $field]]
}

proc validate_email { email } {
	# validate the email address, nothing is also valid
	if {$email == ""} { return 1 }
	return [regexp {^(.+)@(.+)$} $email]
}

proc set_no_trace { var value window } {
	# build a list of traces to delete
	# we only delete traces that start with the command "NULL"
	# and are for the same window as the one we are setting; 03-04-2001
	# all traces which we don't want executed will be defined with the NULL
	# command at the start to identify themselves as "delete me"

	set null_string "NULL $window"
	set null_length [expr {[string length $null_string] - 1}]

	set traces {}
	foreach trace [trace vinfo $var] {
		if {[string range [lindex $trace 1] 0 $null_length] == $null_string} {
			lappend traces $trace
		}
	}

	# remove existing traces
	foreach trace $traces {
		trace vdelete $var [lindex $trace 0] [lindex $trace 1]
	}

	# set the variable to the desired value
	set $var $value

	# reset/add back the original traces
	foreach trace $traces {
		trace variable $var [lindex $trace 0] [lindex $trace 1]
	}
}

#====================================================================#

proc set_focus { w {action ""} } {
	global system
	set top [top_window $w]

	# if focus not set yet then set it to the first field
	# return 1 if set
	if {$system($top,focus) == "" || $action == "force"} {
		set system($top,focus) $w
		return 1
	}

	return 0
}

proc top_window { w } {
	# return the top "window" for the specified widget
	return ".[lindex [split $w .] 1]"
}

#====================================================================#

proc entry_validation { type w event command string new_value } {
	global system

	# if the field is a date or number then attempt to update the field value
	# date field? store the date value
	if {$type == "date"} {
		set ns $system($w,ns)
		set field $system($w,field)
		set date [string trim $new_value]

		# store the original date
		if {[catch { set system($w,original_date) }]} {
			set system($w,original_date) [set $ns\::$field]
		}

		# if the date field is valid set the field (only update on "focusout")
		set date_valid [date::validate $date 0 0 $system($w,original_date)]

		if {$date_valid} {
			set_no_trace $ns\::$field [date::store $date $system($w,original_date)] $w
		}
	} elseif {$type == "time"} {
		set ns $system($w,ns)
		set field $system($w,field)
		set time [string trim $new_value]

		# if the date field is valid set the field (only update on "focusout")
		set time_valid [date::validate_time $time]

		if {$time_valid} {
			set_no_trace $ns\::$field [date::store_time $time] $w
		}
	} elseif {$type == "number"} {
		if { $system($w,blankok) == 1 && $new_value == ""} {
			set number ""
		} else {
			set number [entry_number $w $new_value]
		}
		set ns $system($w,ns)
		set field $system($w,field)
		set_no_trace $ns\::$field $number $w
	}

	# key? validate the insertion
	# other event? ok
	if {$event == "focusout"} {
		# date field? expand the date field on screen if valid
		if {$type == "date"} {
			set ns $system($w,ns)
			set field $system($w,field)
			# if the date field is valid set the field & update the display
			if {$date_valid} {
				$w configure -validate none
				$w delete 0 end
				$w insert end [date::display [set $ns\::$field]]
				$w configure -validate all
			}
		} elseif {$type == "time"} {
			set ns $system($w,ns)
			set field $system($w,field)
			# if the date field is valid set the field & update the display
			if {$time_valid} {
				$w configure -validate none
				$w delete 0 end
				$w insert end [date::store_time [set $ns\::$field]]
				$w configure -validate all
			}
		} elseif {$type == "number"} {
			set ns $system($w,ns)
			set field $system($w,field)

			$w configure -validate none
			$w delete 0 end
			$w insert end [set $ns\::$field]
			$w configure -validate all
		}
		return 1
	} elseif {$event != "key"} {
		return 1
	}

	# only interested in "insert (1)" events
	if {$command != 1} {
		return 1
	}

	# check if there is a size restriction on the field?
	if {![catch { set size $system($w,size) }]} {
		if {$size != ""} {
			if {[string length $new_value] > $size} {
				bell
				return 0
			}
		}
	}

	# check if there is a set of valid entries?
	if {![catch { set valid $system($w,valid) }]} {
		if {$valid != ""} {
			# process each character in the inserted string (are they valid?)
			for {set i 0} {$i < [string length $string]} {incr i} {
				if {[string first [string range $string $i $i] $valid] == -1} {
					bell
					return 0
				}
			}
		}
	}

	# do we allow singlequotes to be entered, the default is no
	if {[catch { set singlequote $system($w,singlequote) } ]} {
		set singlequote 0
	}

	# we are purposely excluding the single quote character from entry as it causes SQL problems
	if {[string first "'" $new_value] != -1 && !$singlequote} {
		bell
		return 0
	}

	# do we allow whitespace to be entered, the default is yes
	if {[info exists system($w,whitespace)] && $system($w,whitespace) == 0} {
		if {[string first " " $new_value] != -1} {
			bell
			return 0
		}
	}

	return 1
}

proc entry_number { w new_value } {
	global system

	set format $system($w,format)
	set number [string trim $new_value]

	# catch any errors processing entered number/expression
	if {[catch {set number [expr {$number + 0}]}]} {
		set number 0
	}

	set number [string trimleft [format "\%$format\x66" $number]]
	return $number
}

#====================================================================#

proc input_phone { wname field title args } {
	set cmd "input_char $wname $field \"$title\" -size 30 -width 30 $args"
	uplevel 1 $cmd
}

proc input_url { wname field title args } {
	set cmd "input_char $wname $field \"$title\" -size 100 -width 50 $args"
	uplevel 1 $cmd
}

proc input_email { wname field title args } {
	set cmd "input_char $wname $field \"$title\" -size 100 -width 35 -email 1 -singlequote 1 -fg darkgreen -disabledforeground darkgreen $args"
	uplevel 1 $cmd
}

#====================================================================#

proc input_date { wname field title args } {
	set cmd "input_char $wname $field \"$title\" -date 1 -size 10 -valid {-/0123456789.} $args"
	uplevel 1 $cmd
}

proc input_time { wname field title args } {
	set cmd "input_char $wname $field \"$title\" -date 2 -size 5 -valid {0123456789:} $args"
	uplevel 1 $cmd
}

#====================================================================#

# wname = name of the window array - see "create_window"
# field = name of the field to store the data
# title = field title (ie: screen prompt)
# args	= options as follows:
#	    -format 10.4	(defines field as 10 before and 4 after ".")
#	    -low value		(lower limit)
#	    -high value 	(upper limit)
#	    -note text		(text to display after number, eg "%"}
#	    -blankok		(value can be an empty string)
# the following notes added by nathan, they may not be 100% correct
#		-var variable_name value of variable will display after number. If the value of the variable changes, so will the text displayed  
#		-customer customer_number
#		-noentry boolean (is the field display only. 1 = Yes, 0 = No)
#		-validate
#		-event
#		-bg
#		-help
#		-help_args

proc input_number { wname field title args } {
	global system setup
	set f [system_new_window $wname]
	set fe $f.c

	set ns [uplevel {namespace current}]
	set system($fe,ns) $ns
	set system($fe,field) $field
	set system($fe,window) $fe
	set system($fe,onchangeX) [set $ns\::$field]

	# define all options
	system_set_defaults $fe required noentry high low {format 10.2} onchange width pluskey \
		note note_config var var_config validate enterkey event labelvar help help_args \
		{align 1} labelwidth label_config {blankok 0}

	# store options and process undefined ones
	set draw_args [system_set_options $fe $args]

	set command "[list draw_input_number $f $fe $title] $draw_args"
	system_input_widget $fe $ns $command $wname

	return $fe
}

proc draw_input_number { f fe title args} {
	global system setup

	set ns $system($fe,ns)

	set labelw [lindex [pc_label $f $title] 1]
	entry $fe -validate all -validatecommand "entry_validation number %W %V %d %S %P"
	set_focus $fe
	pack $fe -side left -anchor nw

	foreach {option value} $args {
		$fe configure $option $value
	}

	# -note? note to display to right of the entry box?
	if {$system($fe,note) != ""} {
		label $f.t -text $system($fe,note) -justify left -font [setup font_small]
		pack $f.t -side left
	}
	if {$system($fe,note_config) != ""} {
		eval "$f.t configure $system($fe,note_config)"
	}

	# -var?
	if {$system($fe,var) != ""} {
		label $f.v -textvariable $ns\::$system($fe,var) -justify left -font [setup font_small]
		pack $f.v -side left

		if {$system($fe,var_config) != ""} {
			eval "$f.v configure $system($fe,var_config)"
		}
	}

	# -enterkey; process event when user tabs/enters out of field
	if {$system($fe,enterkey) != ""} {
		# note: the -enterkey option validates the screen before action
		# use: -event {Tab command} for non-validation
		set cmd "if \{\[validate_window $fe\]\} \{ $system($fe,enterkey);break \} else \{ break \}"
		::event::bind $fe <Tab> +[list namespace eval $ns $cmd]
	}
	
	# -pluskey; as for -enterkey, but only enabled by setup option enable_plus_key
	if {$system($fe,pluskey) != "" && [setup enable_plus_key 0]} {
		# note: the -pluskey option validates the screen before action
		set cmd "if \{\[validate_window $fe\]\} \{ $system($fe,pluskey);break \} else \{ break \}"
		::event::bind $fe <KeyPress-plus> +[list namespace eval $ns $cmd]
	}
	
	# -event? generic event handling: -event {event_name command ...}
	if {$system($fe,event) != ""} {
		foreach {event command} $system($fe,event) {
			# execute the specified command when the event occurs
			::event::bind $fe <$event> "+if \{\[[list namespace eval $ns $command]\]\} break"
		}
	}

	# -labelvar?
	if {$system($fe,labelvar) != ""} {
		$labelw configure -textvariable $ns\::$system($fe,labelvar)
	}

	# -label_config?
	if {$system($fe,label_config) != ""} {
		eval "$labelw configure $system($fe,label_config)"
	}

	# -format? update value to correct details for numerics (.0?)
	if {[string first "." $system($fe,format)] == -1} {
		append system($fe,format) ".0"
	}

	# -align 0? this option removes the widget from the packing/alignment list (see proc pack_widgets)
	if {!$system($fe,align)} {
		array unset system $labelw,pack
	}
	# -labelwidth? set the label width to the specified size, used in conjunction with "-align 0"
	if {$system($fe,labelwidth) != ""} {
		$labelw configure -width $system($fe,labelwidth)
	}

	# -help text?
	if {$system($fe,help) != ""} {
		set cmd [list popup_help $system($fe,help) $fe]
		# append all help arguments onto the command; help_args might be used for "-time 0" or other, see support.tcl - popup_help
		foreach arg $system($fe,help_args) { lappend cmd $arg }
		::event::bind $fe <FocusIn> $cmd
	}

	# -onchange command?
	if {$system($fe,onchange) != ""} {
		set system($fe,calc) "input::check_onchange $fe"
		set_system $fe calc
	}

	input_number_window $fe

	# override the default width?
	if {$system($fe,width) != ""} {
		$fe configure -width $system($fe,width)
	}

	# allow entry of digits & basic expression stuff
	set system($fe,valid) { ()*+-/.0123456789}

	# pop-up menu accessible via the right-mouse-click
	::event::bind $fe <ButtonPress-3> "application_popup_entry $fe %X %Y"

	set_system $fe noentry

	set system($fe,drawn) 1
}

proc update_number { fe name1 name2 op } {
	global system

	set ns $system($fe,ns)
	set field $system($fe,field)
	set format $system($fe,format)

	if {[info commands $fe] != ""} {
		set number [set $ns\::$field]
		if {$number == "" && $system($fe,blankok) == 0} {
			set number 0
		}
		if { $number != "" } {
			# round the number using internal procedure first, to avoid rounding errors
			set dp [lindex [split $format "."] 1]
			if { $dp != "" } { set number [system::round $number $dp] }
			set number [string trimleft [format "\%$format\x66" $number]]
		}
		set state [$fe cget -state]
		set selection [$fe selection present]
		$fe configure -state normal
		$fe delete 0 end
		$fe insert end $number
		$fe configure -state $state

		# if the field was selected before then re-select entire contents
		if {$selection} {
			$fe selection range 0 end
		}
	}
}

proc input_number_window { f } {
	global system
	set format $system($f,format)
	set ns $system($f,ns)
	set field $system($f,field)
	set window $system($f,window)

	set width [system::get_num_width $format]

	$window configure -width $width -justify right

	set number [set $ns\::$field]
	if {$number == "" && $system($f,blankok) == 0} {
		set number 0
	}
	if { $number != "" } {
		# round the number using internal procedure first, to avoid rounding errors
		set dp [lindex [split $format "."] 1]
		if { $dp != "" } { set number [system::round $number $dp] }
		set number [string trimleft [format "\%$format\x66" $number]]
	}
	$window insert end $number

	# set write trace to update variable on screen if changed
	set cmd "NULL $f;update_number $f"
	trace variable $ns\::$field w $cmd
	::event::bind $window <Destroy> +[list trace vdelete $ns\::$field w $cmd]
}

#====================================================================#

# wname  = name of the window array - see "create_window"
# args	 = options as follows:
#   -buttons {name action misc} (also list, eg: {{1 ""} {2 ""}} etc
#	    name   = button text
#	    action = command to action when button is selected
#	    misc   = "none" by default, "novalidate" = no validation (ie: active)
#	    image = image to use (override default)
#   -actions {} same as the "buttons" option except the window is not closed!
#   -maint {record new? [delete_cmd]}
#	    this is a quick way to have generic save, delete, cancel buttons
#	    record = the record name
#	    new?   = 0/1 if new record (ie: disable delete button)
#	    delete_cmd = command to execute if delete button pressed (optional)
#   -orient h/v : horizontal (default) or vertical
#   -progress 0/1 : create progress bar

proc input_button { wname args } {
	global system

	set b [system_new_window $wname]
	set ns [uplevel {namespace current}]

	set command "[list draw_input_button $b $wname $ns] $args"
	system_input_widget $b $ns $command $wname

	return $b
}

proc draw_input_button { b wname ns args } {
	global system setup

	frame $b
	set buttons 0
	pack $b -fill x

	set orient "left"
	set fill "none"
	set takefocus 1

	# process options
	foreach { option value } $args {
		switch -- $option {
			-orient {
				if {$value == "v"} {
					set orient top
					set fill "x"
				} else {
					set orient $value
					set fill "none"
				}
			}
			-takefocus {
				set takefocus $value
			}
			-buttons {
				foreach button $value {
					set bx $b.b[incr buttons]

					set button_args [lrange $button 2 end]
					array set button_options [process_button_args $button_args]

					# title2 is title with underscores removed
					set title [lindex $button 0]
					regsub -all "_" $title "" title2

					set cmd [lindex $button 1]

					if { $button_options(instrument) } {
						set cmd "instrument_button \"$title2\"; $cmd"
					}

					if { [setup delayed_draw 0] && $button_options(delayclose) } {
						set cmd "close_window $wname -delay;$cmd"
					} else {
						set cmd "close_window $wname;$cmd"
					}

					if { !$button_options(novalidate) } {
						set cmd "if \{\[validate_window $b\]\} \{ $cmd; progress::finish_local \}"
					}

					# determine whether we use an image button or standard?
					# if an image exists with the first word then assume image
					if {$button_options(image) == ""} {
						set name [string tolower [lindex $title2 0]]
						if {[make_image $name]} {
							set name image_$name
						} else {
							set name image_unknown
						}
					} else {
						set name image_$button_options(image)
					}

					# special "none"?
					if {$name == "image_none"} { set name "" }

					eval "[list pc_button $bx $name $title [list namespace eval $ns $cmd]] $button_options(tk_args)"
				
					if {!$takefocus} { $bx configure -takefocus 0 }

					pack $bx -side $orient -padx 4 -pady 2 -anchor w -fill $fill

					if { $button_options(report_info) != "" } {
						::event::bind $bx <ButtonPress-3> "application_popup_entry $bx %X %Y REPORT_BUTTON \"$title2\" \"$button_options(report_info)\""
					} else {
						::event::bind $bx <ButtonPress-3> "application_popup_entry $bx %X %Y BUTTON \"$title2\""
					}
				}
			}
			-actions {
				foreach button $value {
					set bx $b.b[incr buttons]
					set cmd "[lindex $button 1]"

					set button_args [lrange $button 2 end]
					array set button_options [process_button_args $button_args]

					# title2 is title with underscores removed
					set title [lindex $button 0]
					regsub -all "_" $title "" title2

					if { $button_options(instrument) } {
						set cmd "instrument_button \"$title2\"; $cmd"
					}

					if { $button_options(trapevents) } {
						set parent $system($b,parent)
						set cmd "event::activate_trapper $parent; $cmd; event::disable_trapper"
					}

					if { !$button_options(novalidate) } {
						set cmd "if \{\[validate_window $b\]\} \{ $cmd; progress::finish_local \}"
					}

					# determine whether we use an image button or standard?
					# if an image exists with the first word then assume image

					if {$button_options(image) == ""} {
						set name [string tolower [lindex $title2 0]]
						if {[make_image $name]} {
							set name image_$name
						} else {
							set name image_unknown
						}
					} else {
						set name image_$button_options(image)
					}

					# special "none"?
					if {$name == "image_none"} { set name "" }
					
					eval "[list pc_button $bx $name $title [list namespace eval $ns $cmd]] $button_options(tk_args)"
					if {!$takefocus} { $bx configure -takefocus 0 }

					pack $bx -side $orient -padx 4 -pady 2 -anchor w -fill $fill

					if { $button_options(report_info) != "" } {
						::event::bind $bx <ButtonPress-3> "application_popup_entry $bx %X %Y REPORT_BUTTON \"$title2\" \"$button_options(report_info)\""
					} else {
						::event::bind $bx <ButtonPress-3> "application_popup_entry $bx %X %Y BUTTON \"$title2\""
					}
				}
			}
			-help {
				set bx $b.b[incr buttons]

				set cmd "instrument_button \"Help\"; "
				append cmd "help_link \"[lindex $value 0]\""

				pc_button $bx image_help "_Help" $cmd
				if {!$takefocus} { $bx configure -takefocus 0 }

				pack $bx -side $orient -padx 4 -pady 2 -anchor w -fill $fill
				::event::bind $bx <ButtonPress-3> "application_popup_entry $bx %X %Y BUTTON \"Help\""
			}
			-maint {
				#deprecated
				# add default buttons & actions for save/delete/cancel
				set record [lindex $value 0]
				set new [lindex $value 1]
				set delete_cmd [lindex $value 2]
				set after_cmd [lindex $value 3]
				set command [string toupper $record]

				# save button; check validation
				pc_button $b.bs image_save "_Save" [list maint_save $b $ns $record $new $wname "$after_cmd"]
				pack $b.bs -side $orient -padx 4 -pady 2

				# delete button; only if maintaining an existing record
				if {$new == 0} {
					if {$delete_cmd == ""} {
						set delete_cmd "if \\\{\\\[confirm $wname \\\"Delete?\\\" -icon warning -default no\\\]\\\} \\\"sql::delete $record;close_window $wname\\\""
					}
					pc_button $b.bd image_delete "_Delete" "namespace eval $ns \"$delete_cmd\""
					pack $b.bd -side $orient -padx 4 -pady 2
				}

				# cancel button; no validation
				pc_button $b.bc image_cancel "Cancel" "namespace eval $ns \"close_window $wname\""
				pack $b.bc -side $orient -padx 4 -pady 2
			}
			-progress {
				# Create progress bar on input buttons, initially not visible
				if {$value} {
					progress::create_local $b
				}
			}
			default {
				$b configure $option $value
			}
		}
	}

	set system($b,drawn) 1
}

# process the args for an individual button, returns an array in list format
# old format was: ?novalidate? ?image name?
# new format is: -option ?value? -option ?value?...
# valid button options:
# -novalidate					don't validate the parent window before evaluating the button script
# -image image_name				image to display on button
# -trapevents					activate trapper before evaluating button script; disable once complete
# -delayclose					for "-buttons": close the window AFTER evaluating the button script
# -tk_args						get passed to the tk button command
# -instrument					record button press
# -report_info					display report info option on right-click menu for this button
# any unrecognised options will be returned as the tk_names value in a list 
proc process_button_args { button_args } {

	set defaults {novalidate 0 image "" trapevents 1 delayclose 0 tk_args {} instrument 1 report_info ""}

	if { [string index $button_args 0] != "-" } {
		array set options $defaults
		if { [lindex $button_args 0] == "novalidate" } {
			set options(novalidate) 1
		}
		set options(image) [lindex $button_args 1]
		set options(tk_args) [lindex $button_args 2]

	} else {
		command::arguments $button_args options {} $defaults
		foreach {option value} [array get options] {

			# check for any unknown arguments and add them to the list of args to pass to Tk
			switch -- $option {
				novalidate -
				trapevents -
				delayclose -
				instrument -
				image -
				report_info -
				tk_args {
					# nothing to do
				}
				default {
					lappend options(tk_args) $option $value
					array unset options $option
				}
			}
		}
	}

	return [array get options]
}

# record the pressing of a button
proc instrument_button { button_name } {
	global setup

	# ignore close and cancel buttons:
	if { [lsearch {"close" "cancel"} [string tolower $button_name]]>=0 } { return }

	# convert the window name and button name to identifier-style format
	set window_name [lindex [lindex $setup(frame_stack) end] 2]
	set eventName [instrument::get_event_name $window_name]
	append eventName "."
	append eventName [instrument::get_event_name $button_name]

	if {$eventName != "."} {
		::instrument::event_count $eventName
	}
}

# Insert or save record, close window
# used by -maint option on input_button
proc maint_save { w ns record new wname after_cmd } {
	# save ok? (validate)
	if {![validate_window $w]} { return }

	# save the record nicely
	# detect duplicate errors (or any other kind also)
	set cmd [expr {$new} ? {"insert"} : {"update"}]
	if {[catch {namespace eval $ns "sql::$cmd $record"} msg]} {
		# check for "duplicate" message? override user message
		if {[string first "duplicate" $msg] != -1} {
			set msg "A record already exists with the details you have entered, please review your entries before trying this again."
		}
		namespace eval $ns [list show_msg $wname $msg -icon error -title "Error during Save"]
	} else {
		if {$after_cmd != ""} {
			namespace eval $ns "$after_cmd"
		}
		namespace eval $ns [list close_window $wname]
	}
}

#====================================================================#

# the input_text widget now defaults the following options:
# -height 4
# -width 50
# -wrap word
# -scroll y

proc input_text { wname field title args } {
	global system setup
	set ns [uplevel {namespace current}]

	set f [system_new_window $wname]

	set ff $f.f
	set ft $ff.t
	set system($ft,ns) $ns
	set system($ft,field) $field

	# define all options
	system_set_defaults $ft lines uppercase lowercase required standard {standard_newline 1} {align 1} {autoselect 0} \
		noentry event {scroll y} note note_config var {expand 0} {tab 1} {enter 0} {fill none} help {address 0} justify

	set draw_args [system_set_options $ft $args]

	set command "[list draw_input_text $f $ff $ft $title] $draw_args"
	system_input_widget $ft $ns $command $wname

	return $ft
}

proc draw_input_text { f ff ft title args} {
	global system setup

	set ns $system($ft,ns)
	set field $system($ft,field)

	set pw [pc_label $f $title]
	frame $ff
	set fx $ff.x
	set fy $ff.y

	# create the text widget
	text $ft -wrap word -height 4 -width 50
	if {[info exists $ns\::$field]} {
		$ft insert end [set $ns\::$field]
	}
	set_focus $ft

	# store options and process undefined ones
	foreach {option value} $args {
		$ft configure $option $value
	}

	# -event? generic event handling: -event {event_name command ...}
	if {$system($ft,event) != ""} {
		foreach {event command} $system($ft,event) {
			# execute the specified command when the event occurs
			::event::bind $ft <$event> "+if \{\[[list namespace eval $ns $command]\]\} break"
		}
	}

	# -align 0? this option removes the widget from the packing/alignment list (see proc pack_widgets)
	if {!$system($ft,align)} {
		set labelw [lindex $pw 1]
		array unset system $labelw,pack
	}

	# display text -note? notes for text appear below the text entry window
	if {$system($ft,note) != ""} {
		label $ff.text -text $system($ft,note) -justify left -font [setup font_small]
		pack $ff.text -side bottom -anchor nw
	}

	# -note_config?
	if {$system($ft,note_config) != ""} {
		eval "$ff.text configure $system($ft,note_config)"
	}

	# the tab key is set to focus to the next field by default, unless -tab 0 is set
	if {$system($ft,tab)} {
		::event::bind $ft <Tab> "+focus \[tk_focusNext %W\]; break"
		::event::bind $ft <Shift-Tab> "+focus \[tk_focusPrev %W\]; break"
	}

	# if only one line is allowed to be entered, bind the enter key same as the tab
	if {$system($ft,lines) != "" && $system($ft,lines) == 1} {
		::event::bind $ft <Return> "+focus \[tk_focusNext %W\]; break"
	}

	# -standard "code" text selection option?
	if {$system($ft,standard) != ""} {
		# press F1 to select from list of options
		# press F2 to maintain list of options (requires level 1 UT security)
		::event::bind $ft <Control-F1> "+NULL"
		# the -standard_newline option is set to 0 when you don't want blank lines inserted between text messages
		::event::bind $ft <KeyPress-F1> "+standard_text_selection $ft $system($ft,standard_newline)"
		set f2 ""
		if {$login::user(UT_SECURITY)} {
			::event::bind $ft <Control-F2> "+NULL"
			::event::bind $ft <KeyPress-F2> "+standard_text_selection $ft maint"
			set f2 "\nPress F2 to maintain messages."
		}
		# add any help to any existing help message (popup)
		set system($ft,help) [string trim "Press F1 to select message.$f2\n$system($ft,help)" \n]
	}

	# -help text?
	if {$system($ft,help) != ""} {
		set cmd [list popup_help $system($ft,help) $ft]
		# append all help arguments onto the command; help_args might be used for "-time 0" or other, see support.tcl - popup_help
		::event::bind $ft <FocusIn> $cmd
	}

	# -scroll x/y?
	if {[string match -nocase "*x*" $system($ft,scroll)]} {
		$ft configure -xscrollcommand "$fx set"
		scrollbar $fx -command "$ft xview" -orient h
		pack $fx -side bottom -fill x
	}
	if {[string match -nocase "*y*" $system($ft,scroll)]} {
		$ft configure -yscrollcommand "$fy set"
		scrollbar $fy -command "$ft yview"
		pack $fy -side right -fill y
	}

	# autoselect? select the text when the cursor focuses into this field (done for address fields by default, removed this logic; 29/11/2003)
	if {$system($ft,autoselect)} {
		::event::bind $ft <FocusIn> "$ft tag remove sel 1.0 end;$ft tag add sel 1.0 end"
	}

	pack $ft -side left -anchor nw -expand [expr {!$system($ft,address)}] -fill $system($ft,fill)
	pack $ff -fill both -expand $system($ft,expand)

	# display text -note? notes for text appear below the text entry window

	if {$system($ft,address)} {
		set map [get_map_name $ns [get_currency_code $ns]]
		# make the image then display the country/map button
		if {[make_image $map]} {
			button $ff.addr -image image_$map -command [list zoomin_address $ft] -font [setup font_small] -takefocus 0
			pack $ff.addr -side left -padx 5
		}
	}

	if {$system($ft,expand)} {
		pack $ft -fill both
		pack [lindex $pw 0] -expand $system($ft,expand)
	}

#	bind $ft <KeyRelease> "+set_text_field $ft"
#	bind $ft <ButtonRelease> "+set_text_field $ft"

	# rename the $ft command, provide a wrapper which then updates the text variable after the text is modified
	rename $ft $ft\_
	proc $ft {command args} "
		# perform the specified command
		set result \[eval uplevel \[list $ft\_ \$command \$args\]\]

		if \{\[string equal \$command insert\] || \[string equal \$command delete\]\} \{
			set_text_field $ft
		\}

		return \$result"

	# display text -var?
	if {$system($ft,var) != ""} {
		label $ff.v -textvariable $ns\::$system($ft,var) -justify left -font [setup font_small]
		pack $ff.v -side left
	}

	# pop-up menu accessible via the right-mouse-click
	::event::bind $ft <ButtonPress-3> "application_popup_entry $ft %X %Y TEXT"

	# set write trace to update variable on screen if changed
	set cmd "NULL $ft;update_text $ns\::$field $ft\_ $ff 0"
	trace variable $ns\::$field w $cmd
	::event::bind $ft <Destroy> +[list trace vdelete $ns\::$field w $cmd]

	# -lines? set-up some logic to restrict the entry length
	set lines $system($ft,lines)
	if {$lines != ""} {
		set cmd "\[input_text_lines $ft $lines\]"
		set system($ft,validate) [list $cmd "The maximum number of lines allowed is $lines.\nPlease truncate this entry."]
	}

	set_system $ft noentry

	set system($ft,drawn) 1
}

proc update_text { field window frame resize name1 name2 op } {
	global system
	global setup

	if {[info commands $window] != ""} {
		set text [set $field]

		set ft $frame.t
		
		set state [$window cget -state]
		$window configure -state normal
		$window delete 1.0 end
		$window insert end $text
		$window configure -state $state
		$window tag add "$system($ft,ns).$frame.text" 0.0 end

		if {$system($ft,justify) != ""} {
			$window tag configure "$system($ft,ns).$frame.text" -justify $system($ft,justify)
		}

		if { $resize } {
			# don't resize if width/height options set in system array (specified by caller)		
			if { $system($frame,text_height) == 0 } {
				set height [text_height $text]
				$window configure -height $height
			}
			if { $system($frame,text_width) == 0 } {
				set width [text_width $text]
				
				# Charlotte: calculate the width for font_small so that it won't resize the entire window according to bigger font size calculation
				set font [$window cget -font]				
				if {$font == [setup font_small]} {
					set width [expr $width*[lindex [setup font_small] 1]/[lindex $setup(font) 1]]
				}
		
				$window configure -width $width
			}
		}
	}
}

proc set_text_field { ft } {
	global system
	set ns $system($ft,ns)
	set field $system($ft,field)

	# get the full text & trim off trailling new line characters
	set text [string trimright [$ft get 1.0 end] "\n"]

	# update the field value
	set_no_trace $ns\::$field $text $ft
}

proc input_text_lines { ft lines } {
	global system
	set ns $system($ft,ns)
	set field $system($ft,field)

	# if text lines more than allowed return 0 (invalid)
	if {[llength [split [set $ns\::$field] "\n"]] > $lines} {
		return 0
	}

	return 1
}

# Return the current customer, supplier or company currency
proc get_currency_code { ns } {
	set currency ""
	if {[namespace tail $ns] == "AP"} {
		catch {set currency [set ${ns}::master(PRICING_CURRENCY)]}
		if {$currency != ""} { return $currency }
	}
	catch {set currency [set ${ns}::master(BALANCE_CURRENCY)]}
	if {$currency != ""} { return $currency }
	catch {set currency [set ${ns}::customer(BALANCE_CURRENCY)]}
	if {$currency != ""} { return $currency }
	catch {set currency [set ${ns}::company(CURRENCY_CODE)]}
	if {$currency != ""} { return $currency }

	return $login::company(CURRENCY_CODE)
}

# Return map name based on phone no or currency code
proc get_map_name { ns currencyCode } {
	set phone ""
	set fax ""
	set mobile ""
	catch {set phone [set ${ns}::master(PHONE_NO)]}
	catch {set fax [set ${ns}::master(FAX_NO)]}
	catch {set mobile [set ${ns}::master(MOBILE_NO)]}
	if {$phone == ""}  { catch {set phone [set ${ns}::customer(PHONE_NO)]} }
	if {$fax == ""}    { catch {set fax [set ${ns}::customer(FAX_NO)]} }
	if {$mobile == ""} { catch {set mobile [set ${ns}::customer(MOBILE_NO)]} }

	# Get countryCode countryName mapID
	set countryInfo [address::guess_country $phone $fax $mobile $currencyCode]
	set map [lindex $countryInfo 2]
	return $map
}

#====================================================================#

proc input_address { wname field title args } {
	set cmd "input_text $wname $field \"$title\" -lines 4 -wrap none -scroll {} -address 1 $args"
	uplevel 1 $cmd
}

proc zoomin_address { ft } {
	global system
	global tcl_platform

	set ns $system($ft,ns)
	set field $system($ft,field)

	# get the full address & trim off trailing new line characters
	set address [string trimright [$ft get 1.0 end] "\n"]

	# URL encoding
	regsub -all "\n" $address "+" address
	regsub -all "," $address "" address
	regsub -all " " $address "+" address
	regsub -all "&" $address "+" address
	regsub -all "\\+\\+" $address "+" address

	# link to mapping website (all google)
	switch -- [get_currency_code $ns] {
		"AU"  { set host "maps.google.com.au" }  
		"AUD" { set host "maps.google.com.au" }  
		"CNY" { set host "ditu.google.cn" }
		"EUR" { set host "maps.google.fr" }  
		"GBP" { set host "maps.google.co.uk" }
		"JPY" { set host "maps.google.co.jp" }
		"NZ"  { set host "maps.google.co.nz" }
		"NZD" { set host "maps.google.co.nz" }
		"RMB" { set host "ditu.google.cn" }
		"STG" { set host "maps.google.co.uk" }
		"UKP" { set host "maps.google.co.uk" }
		"YEN" { set host "maps.google.co.jp" }
		default { set host "maps.google.com" }
	}
	set url "http://$host/maps?f=q&hl=en&geocode=&q=$address"

	# On Win95, even if there is a web browser installed, it cannot
	# open an internet address, only a local .html file.

	# Win98 doesn't like "/"
	if {[lindex [array get ::tcl_platform] 1] == "4.10"} {
		regsub -all "/" $url "\\\\" url
	}

	# Substitute & with ^&
	# needs this for XP (which is 5.1)
	if {[lindex [array get ::tcl_platform] 1] > "4.10"} {
		regsub -all "&" $url "^&" url
	}

	# Open url (in browser)
	if {$tcl_platform(platform) == "unix"} {
		eval exec [auto_execok firefox] [list $url] &
	} else {
		eval exec [auto_execok start] [list $url] &
	}
}

#====================================================================#

# this command is used to display labels & text on the window
# wname = window name
# title = label of the text
# args	= options, note: options before the -text option are for the
#	  labeledwidget and after the -text option for the text widget
# -text = text message to display in the label window
# returns the Tk path of the frame containing the text

proc show_text { wname text args } {
	global system
	set f [system_new_window $wname]
	set ft $f.t
	set ns [uplevel {namespace current}]

	set system($ft,ns) $ns
	set system($ft,wname) $wname
	set system($f,wname) $wname
	set system($ft,parent) [system_parent_window $f $ns]

	# define all options
	system_set_defaults $f scroll {expand no} variable {anchor nw} event {padx 4} {pady 2} {text_width 0} {text_height 0} justify
	system_set_defaults $ft justify

	# remove trailing blank lines from the text
	set text [string trimright $text "\n"]

	# store options and get undefined ones
	set draw_args [system_set_options $f $args]
	
	set command "[list draw_text $f $ns $text] $draw_args"
	system_input_widget $f $ns $command $wname

	return $f
}

proc text_height { text } {
	set lines [split $text "\n"]
	set height [llength $lines]
	return $height
}

proc text_width { text } {
	set lines [split $text "\n"]
	set max_width 0
	foreach line $lines {
		set line_width [string length $line]
		if { $line_width > $max_width } {
			set max_width $line_width
		}
	}
	
	set width [expr {$max_width + 5}]
	return $width
}

proc draw_text { f ns text args } {
	global system
	global setup
	

	set height [text_height $text]
	if { $system($f,text_height) != 0 } {
		set height $system($f,text_height)
	}

	set width [text_width $text]
	if { $system($f,text_width) != 0 } {
		set width $system($f,text_width)
	}
	
	# Charlotte: calculate the width for font_small so that it won't resize the entire window according to bigger font size calculation
	foreach {option value} $args {
		if {$option == "-font"} {
			if {$value == [setup font_small]} {
				set width [expr $width*[lindex [setup font_small] 1]/[lindex [setup font] 1]]
			}
		}
	}
	
	frame $f -bd 0 -highlightthickness 0

	text $f.t -wrap word -bd 2 -relief flat -height $height -width $width -bg [$f cget -background]
	$f.t insert end $text
	$f.t configure -state disabled
	$f.t tag add "$ns.$f.text" 0.0 end

	foreach {option value} $args {
		switch -- $option {
			default {
				$f.t configure $option $value
			}
		}
	}

	if {$system($f,justify) != ""} {
		$f.t tag configure "$ns.$f.text" -justify $system($f,justify)
	}

	# -variable? set trace on variable to update the text widget
	if {$system($f,variable) != ""} {
		set cmd "NULL $f.t;update_text $ns\::$system($f,variable) $f.t $f 1"
		trace variable $ns\::$system($f,variable) w $cmd
		::event::bind $f.t <Destroy> +[list trace vdelete $ns\::$system($f,variable) w $cmd]
	}

	# scroll? y only at present
	if {$system($f,scroll) == "y"} {
		$f.t configure -yscrollcommand "$f.y set"
		scrollbar $f.y -command "$f.t yview"
		pack $f.y -side right -fill y
	}

	# -event? generic event handling: -event {event_name command ...}
	if {$system($f,event) != ""} {
		foreach {event command} $system($f,event) {
			# execute the specified command when the event occurs
			::event::bind $f.t <$event> [list namespace eval $ns $command]
		}
	}

	# pop-up menu accessible via the right-mouse-click
	::event::bind $f.t <ButtonPress-3> "application_popup_entry $f.t %X %Y TEXT"

	pack $f.t -anchor $system($f,anchor) -padx 0 -pady 0 -fill both -expand yes

	pack $f -anchor $system($f,anchor) -padx $system($f,padx) -pady $system($f,pady) -fill both -expand $system($f,expand)

	set system($f,drawn) 1
}

# create miscellaneous widgets
# provide pack arguments as value for 
proc create_widget { wname widget args } {
	global system

	initialise_widgets
	set w [system_new_window $wname]
	set ns [uplevel {namespace current}]

	set system($w,wname) $wname
	set system($w,ns) $ns

	system_set_defaults $w {pack {}} {bind {}}
	set draw_args [system_set_options $w $args]

	set command "[list draw_widget $w $widget] $draw_args"
	system_input_widget $w $ns $command

	return $w
}

proc draw_widget { w widget args} {
	global system

	set command "[list $widget $w] $args"
	eval $command

	set bind_args $system($w,bind)
	foreach {cause effect} $bind_args {
		::event::bind $w <$cause> $effect
	}

	set pack_args $system($w,pack)
	set pack_command "[list pack $w] $pack_args"
	eval $pack_command
}

# evaluate the commands that draw the window
proc draw_window_widgets { w {frame ""} } {
	global system setup delayed_window_closes

	# close any windows that are delayed
	close_delayed_windows

	if { [array names system "$w,draw_commands"] == {} } { return }

	set commands_to_evaluate $system($w,draw_commands)
	set unevaluated_commands {}
	# remove the commands in the system array for now, so they don't get evaluated twice
	set system($w,draw_commands) {}

	foreach command_list $commands_to_evaluate {
		set command [lindex $command_list 0]
		if { $frame != "" } {
			set command_frame [lindex $command_list 1]
			# if the widget to be drawn is not on the selected frame (which hasn't been drawn yet), don't draw it
			if { $frame != $command_frame && ![winfo exists $command_frame] } {
				lappend unevaluated_commands $command_list

				continue
			}
		}

		# evaluate the command that will draw the widget
		set start [clock clicks]
		eval $command

		if { [setup log_draw 0] } {
			if { [regexp {^draw_input_button} $command] } { continue }
			set duration [expr {([clock clicks] - $start) / 1000}]
			if { $duration > 100 } { logToFile "$duration ms to evaluate: '$command'" }
		}
	}

	set system($w,draw_commands) $unevaluated_commands
}

proc close_delayed_windows {} {
	global system delayed_window_closes

	foreach delayed_window $delayed_window_closes {
		foreach {ns wname action} $delayed_window {}
		
		set command [list close_window $wname -delayed]
		if { $action != "" } {
			lappend command -action $action
		}

		namespace eval $ns $command
	}
	set delayed_window_closes {}
}

#====================================================================#

# wname = name of the window array - see "create_window"
# Options:
# "-calc"         process the specified command after every key/button
# "-setfocus"     set to 0 to suppress the initial focus (see support/output)
# "-destroy"      proc/script called on destruction of the window ...
# "-packforget"   hide frame and its contents (for subframe, not toplevel)
# "-packremember" undo packforget
# "-redraw"       proc called when window is redrawn (when child window destroyed)
# "-playback"     replay queued events (default true)
# If you use -destroy, -packforget or -packremember, call input_window again to draw the window
proc input_window { wname args } {
	global system setup
	set ns [uplevel {namespace current}]
	# get the parent window (frames may have {child parent} list)
	set w $system($ns,$wname)
	if {[lindex $w 1] != ""} {
		set w [lindex $w 1]
	}

#	event::disable_trapper

	# draw the fields
	if { $setup(delayed_draw) } {

		# if this window/widget has a parent, only draw if the parent is drawn. Otherwise, add to parent's draw list
		if { [info exists system($w,parent)] && $system($w,parent) != $w } {

			set parent $system($w,parent)
			if { $system($parent,drawn) } {
				event::disable_trapper
				draw_window_widgets $parent $w
			} else {
				set draw_command "[list input_window $wname -recursive 1] $args"

				# input_window command has to be evaluated in the correct (caller's) namespace:
				set command "[list namespace eval $ns $draw_command]"
				lappend system($parent,draw_commands) [list $command $w]
				return
			}
		} else {
			event::disable_trapper
			draw_window_widgets $w
		}
	}

	set toplevel [winfo toplevel $w]
	set top [top_window $w]

	# Save stack info, for right-click "Stack Trace"
	set system($w,stack) [get_stack_trace]

	# define all options
	system_set_defaults $w packforget packremember calc destroy {setfocus 1} redraw {playback 1}

	set recursive 0

	# read all options into the usual system($w,*options*) array
	foreach {option value} [system_set_options $w $args] {
		if { $option == "-recursive" && $value != 0 } {
			set recursive 1
			continue
		}
		$w configure $option $value
	}

	# -destroy? simply return now, see "close_window"
	if {$system($w,destroy) != ""} {
		# set alternate array name: destroy_proc (otherwise gets cleared by
		# next call to input_window)
		set system($w,destroy_proc) $system($w,destroy)
		return
	}

	# -packforget? remove the window/frame from the display
	if {$system($w,packforget) != ""} {
		pack forget $w
		return
	}

	# -packremember? put the window/frame back on the display
	if {$system($w,packremember) != ""} {
		pack $w -expand 1 -fill both
		return
	}

	# -redraw? specify command to execute when the screen is redrawn (by closing any child windows), see "select"
	if {$system($w,redraw) != ""} {
		append system($w,redraw_command) "$system($w,redraw);"
	}

	# call the "close_window" procedure when the window is destroyed (via "X")
	if {[winfo class $w] == "Toplevel"} {
		wm protocol $w WM_DELETE_WINDOW [list namespace eval $ns close_window $wname "destroy"]
	}

	# save the "-setfocus" option here
	# we currently only use it for the print dialog and the variables get unset by the "close_window" function
	# after the "update" command included below (problems working with this on 17/05/01)
	set setfocus $system($w,setfocus)

	set_system $w calc

	pack_widgets $w
	update_window $top

	# centre the window for display
	# note: subsequent calls to "input_window" will centre window again
	update idletasks

	# check that the window still exists, we do this here/now because the "update" command above could have
	# triggered an event that closes the window before it's been displayed, eg: user presses Escape
	if {![winfo exists $top]} {
		return
	}

	catch {
		if {$system($w,centre)} {
			centre_window $w
		}
	}

	# display the window (if withdrawn)
	if {[wm state $toplevel] == "withdrawn" && !$recursive} {
		wm deiconify $toplevel
	}

	# focus on the first field in the window, select the contents if active Entry field
	set focus $system($top,focus)
	if {$focus != "" && $setfocus} {
		update
		if {[winfo exists $focus]} {
			if {[lsearch {Frame Labelframe} [winfo class $focus]] != -1} {
				set state "normal"
			} else {
				set state [$focus cget -state]
			}

			focus $focus

			# if the field is not "normal" then focus on the next field
			if {$state != "normal" && !$recursive} {
				# display the window, update then "Tab" to next window
				# don't use the event generate function, this can screw things up with the wrong events

				update

				# re-check the window exists (may not after "update")
				if {[winfo exists $focus]} {
					focus [tk_focusNext $focus]
				}
			}
		}

	}

	if { $recursive } { 
		return 
	}

	# check the window still exists in the system array (it may not due to the 'update' above)
	# playback the queued events
	if { [info exists system($w,playback)] && $system($w,playback) == 1 } {
		after idle [list event::playback $w]
	}
}

# Return execution stack as list of command & argument
proc get_stack_trace {} {
	# Show variables on the stack
	#for {set i [expr [info level] -2]} {$i > 0} {incr i -1} {
	#	set varList [uplevel #$i {info vars}]
	#	logToFile "varList $i $varList"
	#}

	set callStack {}
	set level [info level]
	for {set i 2} {$i < $level} {incr i} {
		set call [uplevel $i {info level 0}]
		# Get fully-qualified command name
		set relative [lindex $call 0]
		if {$relative != ""} {
			set absolute [uplevel $i namespace which -command $relative]
			set arguments [lrange $call 1 end]
			lappend callStack [list $absolute $arguments]
		}
	}
	return $callStack
}

#====================================================================#

# Find max width of all labels in the window
# Get width in pixels, then convert to chars
# This is current used to make the width the same, for all entry fields on a window
# For a notebook, this should be per-tab, not per entire notebook ...
proc pack_widgets { w } {
	global system
	set maxwidth 1

	set windows [array names system $w.*,pack]

	# process each window, if it's a label then check width
	foreach name $windows {
		set w [lindex [split $name ","] 0]

		set width [winfo reqwidth $w]
		if {[expr {$width > $maxwidth}]} {
			set maxwidth $width
		}
	}

	# temporary width setting
	# should divide by width (pixels) of average character
	set maxwidth [expr {1 + (int($maxwidth / 8))}]

	foreach name $windows {
		set w [lindex [split $name ","] 0]
		$w configure -width $maxwidth
	}
}

#====================================================================#

# this procedure is used to determine if a window exists

proc window_exists { wname } {
	global system
	set ns [uplevel {namespace current}]

	# get the parent window (frames may have {child parent} list)
	if {[array names system $ns,$wname] != ""} {
		return 1
	}

	return 0
}

#====================================================================#

# Close a window - remove displayed window and destroy window object
# Usually called when saving an object, eg. after maintaining the object
# close_window supports two formats of passing arguments
# Historic code uses the old format, the new format is preferred

# args -- old format:
# wname - window name
# second optional parameter, 'action'
# can either be destroy or empty string
#
# New format:
# wname - window name
# Options:
# -action, same valid values as above
# -delay, 1 or 0 (if no value is specified, default is 1)
# if delay = 1, close_window will be delayed until either a new window is drawn, or another existing window is closed

proc close_window { wname args } {
	global system delayed_window_closes
	set ns [uplevel {namespace current}]

	set action ""
	set delay 0
	set delayed 0

	if { [string index $args 0] != "-"} {
		set action [lindex $args 0]
	} else {
		foreach option_value [split $args -] {
			if { $option_value == "" } { continue }
			set option [lindex $option_value 0]
			set value [lindex $option_value 1]

			switch -- $option {
				action {
					set action $value
				}
				delay - delayed {
					if { $value != 0 } {
						set $option 1
					}
				}
				default {
					error "Invalid option \"$option\", value = \"$value\""
				}
			}
		}
	}

	# get the parent window (frames may have {child parent} list)
	if {[catch { set w $system($ns,$wname) } ]} {
		# failure to close window, caused by window problem, opening multiple windows of same name?
		set focus [focus]
		if {$focus == ""} { set focus . }
		set top [winfo toplevel $focus]
		set title [wm title $top]
		log "System error on \"close_window $wname $action\" $title."
		return
	}

	# check if we delay the closing of this window (to prevent the user from interacting with the window below)
	if { [setup delayed_draw 0] && $delay } {
		
		lappend delayed_window_closes [list $ns $wname $action]

		# activate a trapper, we don't want events being processed on this window:
		if { !$event::limbo } { ::event::activate_namespace_trapper $wname $ns }
		return
	}

	# now close any windows that have been delayed
	if { !$delayed } {
		close_delayed_windows
	}

	::event::disable_trapper -window $w
	::event::clear_queue

	if {[lindex $w 1] != ""} {
		set w [lindex $w 1]
	}

	if {[info commands $w] != ""} {
		# does the window have a -destroy procedure?
		if {$action == "destroy"} {
			if {![catch { set destroy_proc $system($w,destroy_proc) }]} {
				if {$destroy_proc != ""} {
					namespace eval $ns $destroy_proc
					return
				}
			}
		}

		# does the parent window have a "redraw_command"?
		if {![catch { set pname $system($w,pname) }]} {
			if {$pname != ""} {
				if {![catch { set pw $system($ns,$pname) }]} {
					# check that the parent window still exists
					# get the redraw command, does it have a value?
					if {![catch { set cmd $system($pw,redraw_command) }]} {
						if {$cmd != ""} {
							after idle [list namespace eval $ns $cmd]
						}
					}

					# perform update on all entry fields, done for goods receipts noentry fields
					update_window $pw
				}
			}
		}

		destroy $w
	}

	return 1
}

#====================================================================#

# this procedure builds a list of procedures/expressions call constantly
# to determine if the various buttons are active or in-active

proc validate_window { w } {
	global system

	set toplevel [winfo toplevel $w]
	set topw [top_window $w]

	set activated_trapper 0
	if { !$event::limbo } {
		event::activate_trapper $topw
		set activated_trapper 1
	}

	set ns $system($topw,ns)
	namespace eval $ns "variable validate_message \"\""

	set valid_field 1
	
	# get a list of all window names that need to be checked
	set names {}
	foreach option {range warning_range high low required valiDATE validate} {

		foreach window_name [array names system $topw*,$option] {
			set name [lindex [split $window_name ,] 0]
			lappend names $name
		}
	}

	# make list of all widgets for the window we are validating, that have one or more check-options
	set names [lsort -unique $names]
	
	# loop through the windows (ie. widgets) and validate all options
	foreach name $names {

		# get some useful fields from the system/window array
		foreach f {ns field title date} {
			if {[catch { set sub_$f $system($name,$f) }]} {
				set sub_$f ""
			}
		}
		
		# set default options
		array set options {range "" warning_range "" high "" low "" required "" valiDATE "" validate ""}
		
		# set the options array with the system variables in values
		foreach option {range warning_range high low required valiDATE validate} {
			if {[catch { set value $system($name,$option) }]} { continue }
			set options($option) $value
		}
		
		# check if a required value is set
		if { $options(required) != "" } {
			if {[set $sub_ns\::$sub_field] == ""} {
				set required [namespace eval $ns "expr \{$options(required)\}"]
				if {$required} {
					set $ns\::validate_message "An Entry is Required"
					set valid_field 0
					break
				}
			}
		}
		
		# process the date range and warning date
		if {$options(range) != "" || $options(warning_range) != "" } {
			
			# get the value the field is set to
			set sub_value [set $sub_ns\::$sub_field]
			
			# get the dates
			set range_low [lindex $options(range) 0]
			set range_high [lindex $options(range) 1]
			set warning_low [lindex $options(warning_range) 0]
			set warning_high [lindex $options(warning_range) 1]
			
			# do some checks
			if {$sub_value > $range_high && $range_high != ""} {
				set $ns\::validate_message "The entered date is too far in advance, latest date can be [date::display $range_high]"
				set valid_field 0
				break
			} elseif {$sub_value < $range_low && $sub_value != "" && $range_low != ""} {
				set $ns\::validate_message "The entered date is too far behind, earliest date can be [date::display $range_low]"
				set valid_field 0
				break
			} elseif {$sub_value > $warning_high && $warning_high != ""} {
				set response [confirm . "The entered date ([date::display $sub_value]) is not in the normal range (later than [date::display $warning_high]).\nIs this date correct?"]
				if {!$response} {
					set valid_field 0
					break
				}
			} elseif {$sub_value < $warning_low && $sub_value != "" && $warning_low != ""} {
				set response [confirm . "The entered date ([date::display $sub_value]) is not within than the normal range (later than [date::display $warning_low]).\nIs this date correct?"]
				if {!$response} {
					set valid_field 0
					break
				}
			}
		}
		
		# run the valiDATE procdure and display the message if required
		if { $options(valiDATE) != "" } {
			set proc [lindex $options(valiDATE) 0]
			set $ns\::validate_message [lindex $options(valiDATE) 1]
			
			# call the validation procedure
			# return 1=ok, 0=error, -1=error/message already displayed
			set valid [namespace eval $ns set __temp $proc]
			if {!$valid == 1} {
				set valid_field 0
				break
			}
		}
		
		if { $options(validate) != "" } {
			set proc [lindex $options(validate) 0]
			set $ns\::validate_message [lindex $options(validate) 1]

			# call the validation procedure
			# return 1=ok, 0=error, -1=error/message already displayed
			set valid [namespace eval $ns set __temp $proc]
			if {!$valid} {
				set valid_field 0
				break
			}
		}
		
		# check if a numeric value is to high
		if { $options(high) != "" } {
			# get the high test number and the entered value
			catch { set high [namespace eval $ns "expr \{$options(high)\}"] }
			set sub_value [set $sub_ns\::$sub_field]
			if {$sub_value > $high} {
				if {$sub_date == "1"} { set high [date::display $high] }
				set $ns\::validate_message "Value is too high!\nMaximum high = $high"
				set valid_field 0
				break
			}
		}
		
		# check if an numberic value is to low
		if { $options(low) != "" } {
			catch { set low [namespace eval $ns "expr \{$options(low)\}"] }
			if {[set $sub_ns\::$sub_field] < $low && $low != ""} {
				if {$sub_date == "1"} { set low [date::display $low] }
				set $ns\::validate_message "Value is too low!\nMinimum low = $low"
				set valid_field 0
				break
			}
		}

		# exit the loop if there is a problem
		if {!$valid_field} { break }
	}

	if { $activated_trapper } {
		event::disable_trapper
	}

	# if the field is invalid then make sure it is visible and place the cursor in it...
	if {!$valid_field} {
		# ensure the window is visible to the user
		check_window_visible $name

		# highlight the entry window with a red background
		# also set the disabledbackground colour (in case the widget is somehow disabled, catch errors, not all widgets support this)
		# name is the tk name of the entry widget (not the list, if any)

		set old_colour [$name cget -background]
		set old_bg ""
		catch { set old_bg [$name cget -disabledbackground] }
		$name configure -background red
		catch { $name configure -disabledbackground red }

		# is this is a combobox widget and the entry field is not displayed
		set combo ""
		set old_combo_colour ""
		set old_combo_bg ""
		if {[info exists system($name,ff)] && [info exists system($name,nameonly)] && $system($name,nameonly) == 1} {
			set combo $system($name,ff)
			set old_combo_colour [$combo cget -background]
			catch { set old_combo_bg [$combo cget -disabledbackground] }
			$combo configure -background red
			catch { $combo configure -disabledbackground red }
		}

		# display the message if there is still one to be displayed
		focus $name
		if {[set $ns\::validate_message] != ""} {
			fb_messageBox -title "Data Validation Error" -message [set $ns\::validate_message] -icon error -parent $toplevel
			# re-focus the window (sometimes the messageBox routine loses it again)
			catch { focus $name }
		}

		# restore normal colours
		catch { $name configure -background $old_colour }
		catch { $name configure -disabledbackground $old_bg }

		if {$combo != ""} {
			catch { $combo configure -background $old_combo_colour }
			catch { $combo configure -disabledbackground $old_combo_bg }
		}
		return 0
	}

	return 1
}

#====================================================================#

proc check_window_visible { window } {
	# first select the window contents (called from <FocusIn> event)
	catch {$window selection range 0 end}

	# check the window path name, if any parent is a tabbed notebook then we must ensure
	# that the correct tab page is made visible, eg: window = .window3.tab3.f0.f2.e1
	# the "tab" represents a tab page, the ".f0" afterwards represents page 0

	set wlist [split $window "."]
	set w ""
	set body_bit 0
	foreach bit $wlist {
		if {$body_bit} {
			# found a tab window, raise the page
			set i [string range $bit 1 end]
			Rnotebook:raise $w $i
			break
		}
		# build up the window name, eg: .window1.tab3
		if {$bit == "body"} {
			set body_bit 1
		} elseif {$bit != ""} {
			append w ".$bit"
		}
	}
}

#====================================================================#

proc update_window { w } {
	global system system_calc system_noentry

	# get the top window, this represents the visible window
	set w [top_window $w]

	# process all calculations from input_char/input_window
	foreach win [lsort [array names system_calc $w*]] {
		# check for errors, the array might not exist after evaluating some commands as they may close an open window
		if {![catch { set value $system_calc($win) }]} {
			set ns $system($win,ns)
			namespace eval $ns $value
		}
	}

	# process all noentry options
	foreach win [lsort [array names system_noentry $w*]] {
		# does the window still exist?
		if {[info commands $win] == "" || ![info exists system_noentry($win)]} { continue }

		set value $system_noentry($win)
		set ns $system($win,ns)
		set field ""
		catch {set field $system($win,field)}

		# store original background colour (if not done already)
		if {[catch { set system($win,normal_bg) } ]} {
			set system($win,normal_bg) [$win cget -background]
		}

		# calculate the -noentry value
		set noentry [namespace eval $ns "expr \{$value\}"]

		# check again the window still exists after evaluating the expression
		if {[info commands $win] == ""} { continue }

		# configure the input window
		# if "field" is blank then assume lookup name field
		if {$noentry} {
			if {$field != ""} {
				# notes: the foreground colour gets overwritten when a field is disabled, look at options database
				if {[$win cget -state] != "disabled"} {
					$win configure -state "disabled" -background lightgray
				}
			} else {
				if {[$win cget -background] != "lightgray"} {
					$win configure -state "disabled" -background lightgray
				}
			}
		} else {
			if {$field != ""} {
				if {[$win cget -state] != "normal"} {
					# set state then background (two separate commands)
					$win configure -state normal
					$win configure -background $system($win,normal_bg)
				}
			} else {
				if {[$win cget -background] != $system($win,normal_bg)} {
					# set state then background (two separate commands)
					$win configure -state normal
					$win configure -background $system($win,normal_bg)
				}
			}
		}
	}

	return {}
}

# Set widget option in the system array
proc set_system { w option } {
	global system system_calc system_noentry

	set value $system($w,$option)

	# don't bother storing values if "0" only
	# if the option contains a value then store in the more specific
	# system array, either system_calc() or system_noentry()
	if {$value != "" && $value != "0"} {
		set system_$option\($w) $value
	}
}

#====================================================================#

proc plural { number } {
	if {$number == 1} { return "" }
	return "s"
}

