
# procedures for handling events in fastbase, specifically user-generated events that occur
# when the GUI is not able to respond directly or immediately. 
# When Fastbase draws a new window, there is a period of time between the parent window
# triggering the event to draw a new window, and the new window appearing. During that interval,
# neither the parent nor the child window has control of the application; an 'event_trapper' queues
# all user-triggered events, and processes them when the child window is ready

namespace eval ::event {

#-----------------------------------------------------------------------------------------------------#
#-------------------------------------# PUBLIC INTERFACE #--------------------------------------------#
#-----------------------------------------------------------------------------------------------------#

# limbo = current state ('window limbo') -- 1 means event trapper is in effect, 0 means window is still active
# trap_events = use event trappers -- set this either from the setup option trap_events or from delayed_draw 
# handle_events = control all events -- set this either from the setup option handle_events or from delayed_draw 
variable limbo
variable last_event
variable trap_events
variable handle_events

proc limbo {} {
	variable limbo
	return $limbo
}

# return the time of the last event
proc last_event {} {
	variable last_event
	return $last_event
}

# create an invisible frame on the parent window, used to trap events while 
# the window is still displayed but not able to process events, 
# e.g. while drawing a new window, redisplaying a list etc.
# store trapper widget in system array
# create bindings for all events
# NOTE that this procedure does not change any of the variables in this namesapace, this is because it 
# will most often be called when another event trapper is active i.e. when we are between windows
proc create_trapper { parent } {
	global system setup
	variable trap_events
	variable keyPresses
	variable keyboardMovements
	variable mousePresses
	variable mouseMovements
	variable destroyEvents
	variable otherEvents

	if { !$trap_events } { return }

	if { [info exists system($parent,eventTrapper)] } {
		logEvent "Trapper already created for $parent ($system($parent,eventTrapper))"
		return
	}

	# trapper exists as a frame on the current window
	set eventTrapper [system_new_window $parent "et" -tk_name 1]
	frame $eventTrapper
	pack $eventTrapper

	# Get current properties, based on command::make_callback
	command::inherit_all property
	set propertyList [array get property]

	# bind *all* events on the trapper, specify the type of event (key, mouse destroy, other) 
	# used to decide whether the event should be queued or ignored

	# bind all keyPress events (i.e. press & release), %K provides the specific key
	foreach keyPress $keyPresses {
		::bind $eventTrapper $keyPress [list ::event::handle %W $keyPress $propertyList {} -trapper -keyString %K]
	}

	# bind keyboard movement events
	foreach event $keyboardMovements {
		::bind $eventTrapper $event [list ::event::handle %W $event $propertyList {} -trapper]
	}

	# bind all mouse presses (i.e. press & release), %b provides the specific button
	foreach mousePress $mousePresses {
		::bind $eventTrapper $mousePress [list ::event::handle %W $mousePress $propertyList {} -trapper -buttonNo %b]
	}

	# bind all mouse movement events
	foreach event $mouseMovements {
		::bind $eventTrapper $event [list ::event::handle %W $event $propertyList {} -trapper]
	}

	# bind all window destroy events (i.e. destroy & escape)
	foreach event $destroyEvents {
		::bind $eventTrapper $event [list ::event::handle %W $event $propertyList {} -trapper]
	}

	# store widget path in system array
	set system($parent,eventTrapper) $eventTrapper
}

proc activate_namespace_trapper { wname {ns ""} } {
	global system

	if { $ns == "" } {
		set ns [uplevel { namespace current }]
	}

	if { ![info exists system($ns,$wname)] } { return }

	set parent [lindex $system($ns,$wname) 0]
	activate_trapper $parent
}

# activate the trapper for parent, trapping the user frocus and storing all user-generated events
# create trapper if none exists (e.g. main FB window)
# focus and grab trapper widget
# disable special events, e.g. X on toplevel, tabbing out of frame
proc activate_trapper { parent } {
	global system
	variable trap_events
	variable trapper
	variable limbo
	variable queue
	variable disabledCommands
	variable originalBindings

	if { !$trap_events } { return }
	if { ![winfo exists $parent] } { return }
	logEvent "activating trapper for $parent"

	# Get current properties, based on command::make_callback
	command::inherit_all property
	set propertyList [array get property]

	# cannot have more than one trapper active at one time
	if { $limbo } {
		if { $trapper(parent) != $parent } {
			logEvent "Cannot activate event trapper -- there is already one activated"
			return
		}
	} else {
		# initialise namespace variables
		set queue {}
		set limbo 1
		array unset trapper
		array set trapper { widget "" parent "" toplevel "" focusnext ""}
		array unset destroyCommands
		array set destroyCommands { parentEscape "" toplevelEscape "" topLevelDelete "" }

		# create a trapper if none exists
		if { [array names system $parent,eventTrapper] == ""} {
			create_trapper $parent
		}

		# store trapper information
		set trapper(widget) $system($parent,eventTrapper)
		set trapper(parent) $parent
		set trapper(toplevel) [winfo toplevel $parent]
		set trapper(focusnext) [focus -displayof $parent]

		# disable various special events
		set disabledCommands(parentEscape) [::bind $trapper(parent) <Escape>]
		set disabledCommands(toplevelEscape) [::bind $trapper(toplevel) <Escape>]
		set disabledCommands(toplevelDelete) [wm protocol $trapper(toplevel) WM_DELETE_WINDOW]
		set trapper(parentCursor) [$trapper(parent) cget -cursor]
	}

	# Prevent closing window while in limbo
	::bind $trapper(parent) <Escape> [list ::event::handle %W <Escape> $propertyList {} -trapper]
	::bind $trapper(toplevel) <Escape> [list ::event::handle %W <Escape> $propertyList {} -trapper]
	wm protocol $trapper(toplevel) WM_DELETE_WINDOW [list ::event::handle %W <Escape> $propertyList {} -trapper]

	# change mouse pointer to hourglass
	$trapper(widget) configure -cursor watch
	$trapper(parent) configure -cursor watch

	# trap the user
	focus $trapper(widget)
	grab $trapper(widget)
}

# disable the currently active trapper (if there is currently one active)
# args: list of -option ?value? pairs
# valid options
# -window: if a window paramater is provided, the trapper is only disabled for that window
# -focus: focus on the specified widget after disabling the trapper
proc disable_trapper { args } {
	global system setup
	variable trap_events
	variable trapper
	variable limbo
	variable disabledCommands
	variable originalBindings

	array set options { window "" focus "" }
	foreach option_value [split $args -] {
		if { $option_value == "" } { continue }
		set option [lindex [split $option_value " "] 0]
		set value [lindex [split $option_value " "] 1]
		switch -- $option {
			window - focus { set options($option) $value }
			default { error "Invalid option: \"$option\" (=\"$value\")" }
		}
	}

	# don't disable if we are not in limbo, or the specified window does not match
	if { !$trap_events } { return }
	if { !$limbo } { return	}
	if { $options(window) != "" && $options(window) != $trapper(parent) } { return }

	# check window still exists
	if { [winfo exists $trapper(parent)] } {
		if { $options(focus) != "" } { set trapper(focusnext) $options(focus) }

		# re-enable disabled events
		::bind $trapper(parent) <Escape> $disabledCommands(parentEscape)
		::bind $trapper(toplevel) <Escape> $disabledCommands(toplevelEscape)
		wm protocol $trapper(toplevel) WM_DELETE_WINDOW $disabledCommands(toplevelDelete)

		grab release $trapper(widget)
		# if this window is a popup window, re-grab it so that it's not possible to interact with the window behind.
		if { [info exists system($trapper(parent),popup)] && $system($trapper(parent),popup) == 1 } {
			# make sure this is indeed the top window
			set top_window [lindex [lindex $setup(frame_stack) end] 0]
			if { $top_window == $trapper(parent) } {
				grab $trapper(parent)
			}
		}

		if { [winfo exists $trapper(focusnext)] } {  
			focus $trapper(focusnext)
		}
		
		# return mouse pointer to original state
		$trapper(parent) configure -cursor $trapper(parentCursor)
	}

	set limbo 0
	progress::finish_local
	logEvent "disabled trapper for $options(window); current = $trapper(parent) (focus => $trapper(focusnext))"

}

proc playback_namespace { wname {ns ""} } {
	global system

	if { $ns == "" } {
		set ns [uplevel { namespace current }]
	}

	if { ![info exists system($ns,$wname)] } { return }

	set parent [lindex $system($ns,$wname) 0]
	event::playback $parent
}

# generate all queued events on window and clear queue
proc playback { window } {
	variable queue
	variable trap_events
	variable limbo
	variable trapper

	if { !$trap_events } { return }
	if { $limbo } { return }

	::event::logEvent "playing events for $window"

	# ignore the first event if it is a key-release (i.e. the second part of a user event)
	if { [regexp {KeyRelease} [lindex $queue 0]] } {
		set queue [lrange $queue 1 end]
	}

	foreach event $queue {

		# window may get destroyed by a played-back event, e.g. destroy, escape etc.
		if { ![winfo exists $window] } { break }

		::event::logEvent "Playing back $event on $window"
		catch {event generate $window $event}
	}

	set queue {}
}

# evaluate a script in a namespace, while trapping events. Returns result of evaulating the script.
proc trap_evaluate { script ns wname } {
	activate_namespace_trapper $wname $ns
	set result [namespace eval $ns [list $script]]
	disable_trapper
	return $result
}

# 'wrapper' command for the widget command's bind option 
proc widget_bind { widget widget_tags event script } {
	variable handle_events

	if { !$handle_events } {
		$widget bind $widget_tags $event $script
		return
	}

	# Get current properties, based on command::make_callback
	command::inherit_all property
	set propertyList [array get property]

	# if script begins with +, the script is added to the list of bindings, otherwise it replaces the existing bind
	if { [string index $script 0] == "+" } {
		set script [string range $script 1 end]
		$widget bind $widget_tags $event "+if \{ \[::event::handle %W $event \{$propertyList\} \{$script\}\] == 3 \} \{ break \}"

	} else {
		$widget bind $widget_tags $event "if \{ \[::event::handle %W $event \{$propertyList\} \{$script\}\] == 3 \} \{ break \}"
	}

	::event::logEvent "bound widget $widget ($widget_tags) $event for script:[$widget bind $widget_tags $event]"
}

# Arrange for script to be called when event happens on widget (or group of widgets)
# Tag can be one widget, all widgets or multiple widgets
# 'wrapper' command for Tk bind to ensure all events are handled by ::event::handle (see below)
proc bind { tag event script } {
	variable handle_events

	if { !$handle_events } {
		::bind $tag $event $script
		return
	}

	# Get current properties, based on command::make_callback
	command::inherit_all property
	set propertyList [array get property]

	# if script begins with +, the script is added to the list of bindings, otherwise it replaces the existing bind
	# Break tells tk/bind not to evaluate any more scripts for this event on this widget (or inherited bindings)
	if { [string index $script 0] == "+" } {
		set script [string range $script 1 end]
		::bind $tag $event "+if \{ \[::event::handle %W $event \{$propertyList\} \{$script\}\] == 3 \} \{ break \}"
	} else {
		::bind $tag $event "if \{ \[::event::handle %W $event \{$propertyList\} \{$script\}\] == 3 \} \{ break \}"
	}

	::event::logEvent "bound $tag $event for script:[::bind $tag $event]"
}

#-----------------------------------------------------------------------------------------------------#
#-------------------------------------# SUPPORT PROCEDURES #------------------------------------------#
#-----------------------------------------------------------------------------------------------------#

proc initialise {} {
	global setup
	variable limbo
	variable last_event
	variable trap_events
	variable handle_events

	variable queue {}			; # queue of events that have been trapped
	variable keyPresses			; # list of all keypress events
	variable keyboardMovements	; # list of all keyboard focus events
	variable mousePresses		; # list of all mouse button events
	variable mouseMovements		; # list of all mouse movements
	variable destroyEvents		; # list of all events that destroy the window
	variable otherEvents		; # list of other events to trap
	variable eventTypes			; # array lookup of the above event types
	variable eventID 0			; # unique identifier for each event (for logging)
	variable logLines			; # list of lines waiting to be written to the log
	variable logID				; # unique identifier for each "after $logEvery" command
	variable logEvery			; # how often the logs are written to file (in ms)

	set limbo 0
	set last_event [clock seconds]
	initialise_widgets

	# Default delayed_draw to whatever fbproxy is set to
	# This is also done in system.tcl
	if {[setup delayed_draw] == ""} { set setup(delayed_draw) [setup fbproxy] }

	set trap_events [setup delayed_draw 0]
	set handle_events [setup delayed_draw 0]

	if { [info exists ::setup(trap_events)] } { set trap_events $::setup(trap_events) }
	if { [info exists ::setup(handle_events)] } { set handle_events $::setup(handle_events) }

	# setup event types -- used when handling trapper-generated events
	array set eventTypes {}

	set keyPresses {}
	foreach keyPress {
		<KeyPress>	<Key>
		<KeyRelease>
	} { 
		set eventTypes($keyPress) "key"
		lappend keyPresses $keyPress
	}

	set keyboardMovements {}
	foreach keyboardMovement {
		<FocusIn>
		<FocusOut>
	} { 
		set eventTypes($keyboardMovement) "focus" 
		lappend keyboardMovements $keyboardMovement
	}

	set mousePresses {}
	foreach mousePress {
		<ButtonPress>	<Button>
		<ButtonRelease>
	} { 
		set eventTypes($mousePress) "mouse" 
		lappend mousePresses $mousePress
	}

	set mouseMovements {}
	foreach mouseMovement {
		<Enter>
		<Leave>
		<Motion>
		<MouseWheel>
	} { 
		set eventTypes($mouseMovement) "movement" 
		lappend mouseMovements $mouseMovement
	}

	set destroyEvents {}
	foreach destroyEvent {
		<Escape>
		<Destroy>
	} { 
		set eventTypes($destroyEvent) "destroy" 
		lappend destroyEvents $destroyEvent
	}

	set otherEvents {}
	foreach otherEvent {
		<Activate>
		<Circulate>
		<Colormap>
		<Configure>
		<Deactivate>
		<Expose>
		<Map>
		<Property>
		<Unmap>
		<Visibility>
	} { 
		set eventTypes($otherEvent) "other" 
		lappend otherEvents $otherEvent
	}

	# Not normally delivered to Tk applications:
	# <CirculateRequest>
	# <ConfigureRequest>
	# <Create>
	# <Gravity>
	# <MapRequest>
	# <Reparent>
	# <ResizeRequest>

	# start the log:
	set logLines {}
	set logEvery 1000
	::event::writeLog

	# "wrap" existing all and tag bindings
	foreach tag { all Entry Text Labelframe Frame Button Label Canvas Menu Scrollbar} {
		foreach event [::bind $tag] {
			set script [::bind $tag $event]
			if { $script != "" } {
				::event::bind $tag $event $script
			}
		}
	}

	# make the Return key emulate the Tab key
	::event::bind Entry <Return> "if \{\[info commands %W\] != \"\"\} \{ event generate %W <Tab> \}"

	::event::bind Entry <Key-Up> "if \{\[info commands %W\] != \"\"\} \{ event generate %W <Shift-Tab> \}"
	::event::bind Entry <Key-Down> "if \{\[info commands %W\] != \"\"\} \{ event generate %W <Tab> \}"

	# check field is visible on screen when it receives focus
	::event::bind Entry <FocusIn> "+check_window_visible %W"
	::event::bind Text <FocusIn> "+check_window_visible %W"

	# bind the "control-A" key to "select all"
	::event::bind Entry <Control-Key-a> "+popup_select_all %W CHAR"
	::event::bind Entry <Control-Key-A> "+popup_select_all %W CHAR"
	::event::bind Text <Control-Key-a> "+popup_select_all %W TEXT"
	::event::bind Text <Control-Key-A> "+popup_select_all %W TEXT"

	# provide these bindings also because the Tk default bindings don't include uppercase letters
	event add <<Cut>> <Control-Key-X>
	event add <<Copy>> <Control-Key-C>
	event add <<Paste>> <Control-Key-V>
	event add <<Undo>> <Control-Key-Z>
	event add <<Redo>> <Control-Key-Y>

	# process any text that is to pasted:
	::event::bind Entry <<Paste>> "::text::paste_entry %W"
	::event::bind Text <<Paste>> "::text::paste %W"
}

# handle the occurence of an event for a particular widget or if we are in limbo, queue the event
# called by the event loop, controlled by ::bind (ie. tk bind) called from event::bind
# return result of script evaluation for tk event handler
# widget - tk name of widget that reported the event
# event - tk event name
# propertyList - properties from when event was bound (usually from input_window)
# script - action to perform
# args - options (see below) and extra values passed by tk
proc handle { widget event propertyList script args } {
	global setup errorInfo
	variable limbo
	variable last_event
	variable trapper
	variable eventID
	variable eventTypes
	variable queuedScripts

	set id [incr eventID]
	set last_event [clock seconds]

	set log_event 0
	if {[setup log_event_events {}] == {} || [lsearch [setup log_event_events] [string range $event 1 end-1]]>=0 } {
		::event::logEvent "$id $widget reported $event ($args); propertyList:\"$propertyList\" script:\"$script\""
		set log_event 1
	}

	# connID (for proxy) and redisID
	array set property $propertyList

	# process options and add any substitutions
	array set options { keyString "" buttonNo "" trapper 0}

	foreach option_value [split $args -] {
		if { $option_value == "" } { continue }
		set option [lindex [split $option_value " "] 0]
		set value [lindex [split $option_value " "] 1]

		switch -- $option {
			keyString - buttonNo {
				set options($option) $value
			}
			trapper {
				if { $value != 0 } { set options($option) 1 }
			}
			default {
				error "Invalid option \"$option\" (= \"$value\")"
			}
		}
	}

	set type ""
	if { [info exists eventTypes($event)] } {
		set type $eventTypes($event)
	}

	# if we are not in limbo, evaluate the script in the global namespace and return
	# evaluate any control commands in the above level (the event loop)
	if { !$limbo } {

		# if the event came from the trapper (this can happen sometimes, should prevent it...)
		# release grab
		if { $options(trapper) } {
			logEvent "Trapper reporting events outside of limbo"
			switch -- $type {
				key - focus - mouse - movement {
					grab release $widget
					if { $event == "<FocusIn>" } {
						catch { focus $trapper(focusnext) }
					}
					#event generate $widget <FocusOut>
					return
				}
				destroy - other {
					# nothing
				}
			}
		}

		# 0 = ok, 1 = error, 2 = return, 3 = break, 4 = continue
		set result [catch { namespace eval :: [list eval $script] } msg]
		if { $log_event } {
			if { $result == 0 } { 
				::event::logEvent "$id evaluated, OK"
			} else {
				::event::logEvent "$id evaluated script: result = $result, msg = $msg\n $script; "
			}
		}

		switch -- $result {
			0 { return 0 }
			1 { 
				# if we got an bad window path name error, and the widget that triggered the event no longer exists,
				# then ignore the error -- probably the parent window has been destroyed by a previous event
				if { [regexp "bad window path name" $msg] && ![winfo exists $widget] } return
				error "$msg\n\nWhile evaluating \"$script\" (bound to event):\n$errorInfo" 
			}
			2 { return 0 }
			3 { return 3 }
			4 { return 0 }
		}
	}

	# in limbo.  Queue the event?
	switch -- $type {
		key {
			if { $event == "<Key>" } { event::check_unlock $options(keyString) }
			set event "<[string range $event 1 end-1]-$options(keyString)>"
			::event::queue $event
		}
		focus { 
			focus $trapper(widget)
			if { $log_event } { ::event::logEvent "$id Recorded focus $event" }
		}
		mouse { 
			if { $log_event } { ::event::logEvent "$id Recorded mouse $event, button $options(buttonNo)" }
		}
		movement {
			$trapper(widget) configure -cursor watch
			grab $trapper(widget)
			if { $log_event } { ::event::logEvent "$id Recorded movement $event" }
		}
		destroy { 
			event::check_unlock "escape"
			if { $log_event } { ::event::queue $event }
		}
		other {
			if { $log_event } { ::event::logEvent "$id Recorded $event" }
		}
		default {
			# we are in limbo, but an event has been reported by a widget other than the trapper
			if { $log_event } { ::event::logEvent "$id Recorded $event from $widget" }
		}
	}

	return 0
}

proc check_unlock { key } {
	variable escapeCount
	variable soFar

	::event::logEvent "unlock $key ?"
	set unlock 0

	if { $key == "" } { return }

	if { [setup limbo_escape_method "escape"] == "escape" } {
		if {![info exists escapeCount]} { set escapeCount 0 }
		::event::logEvent "escape count $escapeCount"
		if { $key == "escape" } {
			incr escapeCount
		} else {
			set escapeCount 0
		}

		if { $escapeCount >= [setup escape_threshold 10] } {
			set escapeCount 0
			event::unlock
		}
	} else {
		if { ![info exists soFar] || $key == "escape"} {
			set soFar ""
		}

		set key [string tolower $key]
		append soFar $key

		set escapeSequence [setup escape_sequence "monkey"]
		if { "$soFar$key" == $escapeSequence } {
			set soFar ""
			event::unlock
		} elseif { ![string match "$soFar$key*" $escapeSequence] } {
			set soFar ""
		}
	}
}


proc unlock {} {
	global tcl_platform setup current_update system procSource
	global user_login_date
	global user_login_time
	variable limbo
	variable unlocked

	event::logEvent "Unlocking Fastbase"

	event::disable_trapper
	set message "Your Fastbase session may be frozen.\nAn email has been sent to Fastbase support. Click Exit to terminate the application."

	# get the user/company details
	set company $login::company(NAME)
	set email $login::user(EMAIL_ADDRESS)
	set name $login::user_name
	if {$email == ""} {
		set email $setup(email_support)
	}
	
	set email_message "Details of this have been emailed to \"$setup(email_notify)\".\nYou should contact your support staff for additional assistance."
	# if no email is setup, don't send email - display message
	if {$setup(smtphost) == ""} {
		set email_message "Details could not be emailed to \"$setup(email_notify)\" because Fastbase email \nhas not been configured. You should contact your support staff for additional assistance."
	}

	# set body of the email, include useful information
	set extra [expr {$setup(remote_host) == ""} ? {"(local)"} : {"(remote host: $setup(remote_host))"}]
	
	set mysql_version ""
	if { [catch { set mysql_version [sql "select version()"] }] } {
		set mysql_version "Unable to obtain version."
	}

	set os $tcl_platform(os)
	set os_version $tcl_platform(osVersion)
	
	set stack_trace [get_stack_trace]

	set window_stack {}
	foreach aname [array names system "*,stack"] {
		set call $system($aname)
		set window [lindex [split $aname ,] 0]
		set command [lindex $call 0]
		set arguments [lindex $call 1]
		set sourceFile ""
		catch {set sourceFile $procSource($command)}
		lappend window_stack "$window called from \"$command $arguments\" (file: \"$sourceFile\")"
	}

	set body "User: $name ([system::user_name $name])\nCompany: $company\nDatabase: $setup(database_name) (v1.$current_update)\nLogged in: $user_login_time $extra\nTcl: [info patchlevel]\nMysql: $mysql_version\nOperating System: $os $os_version\nSoftware release: [software_release_name]\n\n"
	append body "Interruption:\n\n$message\n\nStack Trace:\n\n[join $stack_trace \n]\n"
	append body "\n\nWindow Stacks:\n\n[join [lsort $window_stack] \n]\n"

	set focus [focus]
	if {$focus == ""} { set focus . }
	set top [winfo toplevel $focus]
	set title [wm title $top]

	set w .unlock
	toplevel $w
	wm title $w "Fastbase Interruption"
	wm transient $w .

	set tw $w.f1
	set mw $w.f2
	set bw $w.f3

	frame $tw -relief groove -bd 2
	frame $mw -relief groove -bd 2
	frame $bw

	text $tw.2 -wrap word -fg red -relief flat -height 3 -font $setup(font_bold) -bg [$w cget -bg]
	$tw.2 insert end $message
	$tw.2 configure -state disabled
	pack $tw.2 -side left -fill x -expand yes

	text $mw.1 -wrap word -height 2 -bg [$w cget -bg] -relief flat
	$mw.1 insert end $email_message
	$mw.1 configure -state disabled
	pack $mw.1 -pady 4 -padx 4 -fill x -expand yes

	pc_button $bw.1 image_continue "Continue" "destroy $w; catch \{ log \"Fastbase interrupted & continued.\"\}"
	pc_button $bw.2 image_firstaid "Exit Fastbase" "login::exit"
	pack $bw.1 $bw.2 -side left -padx 4 -pady 4

	pack $tw $mw $bw -side top -fill x -expand yes -anchor w -padx 2 -pady 2

	# position the error window
	wm geometry $w "+50+50"
	focus -force $bw.1

	# display error window and signal sound
	update
	grab $w
	bell

	# log all errors in full to the as.log file always
	# some errors can't be logged - usually when the SQL server has done something (lost connection etc)
	catch { log "Fastbase Interruption in '$title'\n$body" }

	# send email message to fastbase
	# catch errors and record a log message
	if {$setup(smtphost) != ""} {
		if {[catch {sendmail $email $setup(email_notify) -alert 1 -subject "FastBase Interruption in '$title'" -body $body} msg]} {
			log "Error encountered sending notification email to '$email' ($msg)"
		}
	}
}

proc startTrace { id {name1 ""} {name2 ""} {op ""} args} {
	global setup
	variable traces
	variable activeTrace
	variable startTraceID

	# set activeTrace to true
	# incr activeTrace
	set activeTrace 1

	set startID [incr startTraceID]
	set variable_name [lindex $traces($id) 0]
	set script [lindex $traces($id) 1]
	set active [lindex $traces($id) 2]
	set traces($id) [list $variable_name $script 1]
	regsub -all \n $script "" logScript

	if { [setup log_draw 0] } { ::event::logEvent "TS$startID $op trace $id activated for $name1 $name2: $logScript" }

}

proc finishTrace { id {name1 ""} {name2 ""} {op ""} args} {
	global setup
	variable traces
	variable activeTrace
	variable finishTraceID

	set finishID [incr finishTraceID]
	set variable_name [lindex $traces($id) 0]
	set script [lindex $traces($id) 1]
	set active [lindex $traces($id) 2]
	set traces($id) [list $variable_name $script 0]
	regsub -all \n $script "" logScript

	if { [setup log_draw 0] } { ::event::logEvent "TF$finishID $op trace $id finished for $name1 $name2: $logScript" }

	# set activeTrace to false
	# incr activeTrace -1
	set activeTrace 0
}

# add an event to the event queue
proc queue { event } {
	variable queue
	variable limbo

	if { !$limbo } { return }

	lappend queue $event

	::event::logEvent "Queueing $event"
}

# discard all queued events
proc clear_queue {} {
	variable queue

	set queue {}
}

proc logAllBindings {} {

	logToFile "Widget bindings:"
	set allTags [logAllChildBindings . 0]

	logToFile "Tag bindings:"
	foreach tag $allTags {
		logToFile "$tag:"
		foreach event [::bind $tag] {
			regsub -all \n [::bind $tag $event] "\n\t   " binding
			logToFile "\t$event\n\t ->$binding"
		}
	}
}

proc logAllChildBindings { window depth } {
	
	set indent [string repeat \t $depth]
	set nextDepth [expr {$depth + 1}]

	set tags {}
	foreach tag [bindtags $window] {
		if { [string index $tag 0] == "." } continue
		lappend tags $tag
	}

	logToFile "$indent$window ('[join $tags "', '"]')"
	foreach event [::bind $window] {
		regsub -all \n [::bind $window $event] "\n$indent   " binding
		logToFile "${indent}$event\n$indent ->$binding"
	}

	set children [winfo children $window]
	foreach child $children {
		set childTags [logAllChildBindings $child $nextDepth]
		eval "lappend tags $childTags"
	}

	return [lsort -unique $tags]
}

proc logEvent { message args} {
	global errorInfo
	variable logLines
	variable limbo
	variable activeTrace
	variable logID

	if { [setup log_event_procs {}] == {} } { return }

	# process options
	array set options { force 0 }

	foreach option_value [split $args -] {
		if { $option_value == "" } { continue }
		set option [lindex [split $option_value " "] 0]
		set value [lindex [split $option_value " "] 1]

		switch -- $option {
			force {
				if { $value != 0 } { set options(force) 1 }
			}
			default {
				error "Invalid option \"$option\" (= \"$value\")"
			}
		}
	}

	# get information
	set now [clock seconds]
	set timestamp [clock format $now -format "%H:%M:%S"]
	set parentProc ""
	set i [expr [info level] - 1]
	if {$i > 0} {
		set call [info level $i]
		set parentProc [lindex $call 0]
	}
	set grandparentProc ""
	set i [expr [info level] - 2]
	if {$i > 0} {
		set call [info level $i]
		set grandparentProc [lindex $call 0]
	}

	set relative_proc [lindex [split $parentProc :] end]
	if { [lsearch [setup log_event_procs {}] $relative_proc] < 0} { return }

	set logString "$timestamp \[$grandparentProc -> $parentProc\] "
	if { $limbo } {
		append logString "LIMBO!! "
	} else {
		append logString "          "
	}

	append logString "$message"
	lappend logLines $logString

	if { $options(force) } { 
		logToFile $logString
	}

	set previousErrorInfo $errorInfo
	# check that writeLog is still active:
	if { [catch {after info $logID}] } {
		set errorMessage "Log thread interrupted; errorInfo:\n$previousErrorInfo"
		set logLines [linsert $logLines end-1 $errorMessage]
		writeLog
	}
}

proc writeLog {} {
	variable logLines
	variable logID
	variable logEvery

	if { [setup log_event_procs {}] == {} } { return }

	if { $logLines != {} } { 
		set now [clock seconds]
		set today [clock format $now -format "%Y-%m-%d"]
		set fileName [file join [setup event_log_filepath "//Moa/public/log/"] "events-$today\.log"]
		if {![catch {set fileId [open $fileName a 0664]}]} {

			foreach logString $logLines {
				puts $fileId $logString
			}
			close $fileId
			set logLines {}
		}
	}

	# log every second
	after cancel ::event::writeLog
	set logID [after $logEvery ::event::writeLog]
}

initialise

}
