
#-------------------------------------------
# (c) Peter Campbell Software 2001-2004, all rights reserved.
# http://www.fastbase.co.nz
#-------------------------------------------

# FastBase remote access client for the accounting system (GUI).
# Load configuration file (c:\fastbase.dat) then connect to the server

# Should use environment for setup, database_name shouldn't be required on command line...

# Necessary only if we don't have the setup toolkit
namespace eval setup {}
proc ::setup::useFile { fileName } {
	set fileName [file normalize $fileName]
	namespace eval :: [list source $fileName]
}

# Get value of setup parameter
proc setup { name {default ""} } {
	global setup

	if {[catch {set value $setup($name)}]} {
		set value $default
		set setup($name) $value
	}
	return $value
}

proc remote_check_setup { name default } {
	global setup

	# if the setup element doesn't exist then set the specified default value
	if {[catch { set setup($name) }]} {
		set setup($name) $default
	}
}

# Perform remote no-op command
proc remote_polling {} {
	global setup

	# Is the fbserver connection idle
	if {$::fbremote::responseID == $::fbremote::requestID} {
		fbremote::command "eval" ""
	}
	if {$::fbremote::disconnected} { offer_reconnect_remote }

	after [expr {$setup(remote_polling) * 1000}] { remote_polling }
}

# Send command to fbserver, wait for response
proc remote_command { command value } {
	if {$::fbremote::disconnected} { offer_reconnect_remote }

	set result [fbremote::command $command $value]
	set status [lindex $result 0]
	if {$status != "OK"} {
		set msg [lindex $result 1] 
		if {$::fbremote::disconnected} { offer_quit_remote $msg }
		error $msg
	}
	return $result
}

# Send sql command to fbserver, wait for response
proc remote_sql { command } {
	global errorInfo

	if {$::fbremote::disconnected} { offer_reconnect_remote }

	if {[catch { set result [fbremote::sql $command] } msg]} {
		if {$::fbremote::disconnected} { offer_quit_remote $errorInfo }
		error $errorInfo
	}
	return $result
}

# Lost remote connection, allow the user to reconnect or exit
proc offer_reconnect_remote {} {
	global setup

	if {[info procs ::fb_messageBox] == ""} { return }

	set boxtext "The connection to the remote server has been lost.\n\
			Click Retry to reconnect to the server, or Cancel to exit.\n\
			If this problem persists, contact Fastbase for assistance."
	set answer [fb_messageBox -title "Lost Connection" -icon error -parent . -type retrycancel -default retry -message $boxtext]
	if {$answer == "cancel"} { exit }

	# Attempt reconnect
	while {[catch {fbremote::connect $setup(remote_host) $setup(remote_user) $setup(remote_password) \
							$setup(remote_database) $setup(remote_port)} msg]} {
		set boxtext "$msg\nClick Retry for another attempt, or Cancel to exit."
		set answer [fb_messageBox -title "Could Not Reconnect" -icon error -parent . -type retrycancel -default retry -message $boxtext]
		if {$answer == "cancel"} { exit }
	}
}

# Lost remote connection, allow the user to exit only
proc offer_quit_remote { msg } {
	if {[info procs ::fb_messageBox] == ""} { return }

	set boxtext "The connection to the remote server has been lost\n$msg\n\
			If this problem persists, contact Fastbase for assistance."
	set answer [fb_messageBox -title "Lost Connection" -icon error -parent . -type ok -message $boxtext]
	exit
}

# Download new version of software, use checksums if possible
proc get_new_version {} {
	global setup

	set setup(remote_file) "Checking for Software Updates"

	# Can we calculate file checksums
	set use_checksum 0
	if {[catch { package require sha1 2.0.3 }]} { set use_checksum 0 }

	set data [remote_command "get_files" $use_checksum]
	set filelist [lindex $data 1]

	# Did the server provide checksums
	set firstFile [lindex $filelist 0]
	if {[llength $firstFile] < 3} { set use_checksum 0 }

	foreach record $filelist {
		set fileName [lindex $record 0]
		set ftime [lindex $record 1]
		set fchecksum [lindex $record 2]

		# Should we download the file
		set download_action ""

		if {[file exists $fileName]} {
			set mtime [file mtime $fileName]

			if {$use_checksum} {
				set digest [get_sha1_digest $fileName]
				if {$digest != $fchecksum} { set download_action "Download" }
			} else {
				if {$mtime < $ftime} { set download_action "Getting" }
			}
		} else {
			set download_action "New File"
		}

		if {$download_action != ""} {
			set setup(remote_file) "${download_action}: $fileName"
			update

			set data [remote_command "get_file" $fileName]

			# Create directory if needed
			set dir [file dirname $fileName]
			if {[catch { file mkdir $dir }]} {
				set setup(remote_file) "Cannot create directory $dir"
				update
				after 500
				continue
			}

			# Create/open file for writing
			if {[catch { set fid [open $fileName w] } msg]} {
				set setup(remote_file) "Cannot open $fileName $msg"
				update
				after 300
				continue
			}

			fconfigure $fid -translation binary
			puts -nonewline $fid [lindex $data 1]
			close $fid

			# set the modification time on the file to the same as the "server"
			catch { file mtime $fileName $ftime }
		}
	}
}

# Return 40-byte hex string, the sha1 checksum of file contents
proc get_sha1_digest { fileName } {
	set fid [open $fileName]
	fconfigure $fid -translation binary
	set data [read $fid]
	set digest [::sha1::sha1 $data]
	close $fid
	return $digest
}

# Draw a "select a server" window
# Called if there is more than one remote server defined
proc select_remote_server {} {
	global setup

	# only one server? select it automatically
	if {[llength $setup(remote_servers)] == 1} {
		return 0
	}

	# environment::standard_libraries will be called in as.tcl
	# source what we need to draw a window
	# create a new fbclient download zip file (see website) ...
	source common/command.tcl
	source support/combobox.tcl
	source widgets.tcl
	source toolkit/event.tcl

	initialise_widgets
	frame .select -borderwidth 3 -relief groove
	pack .select

	label .select.l1 -text "FastBase Remote Access Client" -font {Arial 14}
	label .select.l2 -text "Select Server" -font {Arial 14}
	pack .select.l1 .select.l2 -padx 6 

	set setup(remote_server) ""

	set ff [combobox::combobox .select.ff -width 35 -editable 0 -takefocus 1 -font {Arial 14} -bg gray95]

	# display button on screen for all servers
	foreach server $setup(remote_servers) {
		set name [lindex $server 0]
		$ff list insert end $name
	}

	# set initial selection to first option
	$ff configure -value [lindex [lindex $setup(remote_servers) 0] 0]

	set b [button .select.fb -text "Connect" -command "set_remote_server $ff" -font {Arial 14 bold} -fg darkred]

	pack $ff $b -pady 5 -padx 6
	pack $b

	# escape key will close window
	::event::bind . <Escape> "set setup(remote_server) \"\""

	# centre window
	wm resizable . 0 0
	update
	set xmax [winfo screenwidth .]
	set ymax [winfo screenheight .]
	set x [expr {($xmax - [winfo reqwidth .]) / 2}]
	set y [expr {($ymax - [winfo reqheight .]) / 2}]
	wm geometry . "+$x+$y"

	focus -force $ff
	tkwait var setup(remote_server)

	# remove the selection frame
	destroy .select
	wm resizable . 1 1

	# destory the binding created above
	::event::bind . <Escape> ""

	# return the selected server
	return $setup(remote_server)
}

proc set_remote_server { ff } {
	global setup

	set value [$ff get]
	set id 0

	# find the index of the selected remote server
	foreach server $setup(remote_servers) {
		set name [lindex $server 0]
		if {[string match $name $value]} {
			set setup(remote_server) $id
			return
		}
		incr id
	}

	set setup(remote_server) ""
}

proc edit_setup_dat {} {
	# this procedure is used to load the remote computers "setup.dat" file and display its contents
	# you can also update the modified file (superuser login only)

	# get the remote "setup.dat" file
	variable setupdat [lindex [remote_command "get_file" "setup.dat"] 1]
	regsub -all "\r" $setupdat "" setupdat
	variable setupold $setupdat

	create_window setupw "" "setup.dat"
	create_frame fwd setupw "Setup File" -expand yes

	set w [input_text fwd setupdat "File:" -height 20 -width 80 -tabs {1c 2c 3c 4c} -wrap none -expand yes -tab 0 -undo 1]

	# set the insert cursor at the top of the code window
	system_widget_command $w mark set insert 1.0

	input_button setupw -actions {{"_Save File" update_setup_dat "" save} {"Cancel" cancel_setup novalidate}}

	input_window setupw -destroy cancel_setup
	input_window setupw
}

proc cancel_setup {} {
	variable setupdat
	variable setupold

	if {[string trimright $setupdat \n] != [string trimright $setupold \n]} {
		if {![confirm setupw "You have made changes to this file.\nAre you sure you want to cancel your changes?" -icon warning -default no]} {
			return
		}
	}

	close_window setupw
}

proc update_setup_dat {} {
	variable setupdat

	remote_command "update_source" [list "setup.dat" [clock seconds] $setupdat]

	show_msg setupw "Done"

	close_window setupw
}

#--------------------------------------------------

# this procedure is accessible from the fastbase "File" menu (superuser login only) to update a remote server

proc remote_software_update {} {
	global setup
	variable select_list {}

	# get list of source files from the server here
	set local_files [get_local_file_list]

	# get list of source files at the remote server
	set data [remote_command "get_files" ""]
	set remote_files [lindex $data 1]

	# build list of files that are missing or different
	variable transfer_list {}

	foreach row $local_files {
		set lname [lindex $row 0]
		set ltime [lindex $row 1]

		set file_not_found 1

		# is the local file on the remote server?
		foreach row $remote_files {
			set rname [lindex $row 0]

			# convert filenames to lowercase when comparing (no case distinction)
			if {[string match -nocase $rname $lname]} {
				set rtime [lindex $row 1]
				# check the file times are the same
				# note: on some boxes the time is set to the time + 1 (as occurs at briteuro with an NT server), not sure why; 04/04/2005
				if {$rtime != $ltime && $rtime != ($ltime + 1)} {
					set cols [list $lname [show_filetime $ltime] [show_filetime $rtime]]
					lappend transfer_list [list $cols $ltime]
				}
				set file_not_found 0
				break
			}
		}

		if {$file_not_found} {
			set cols [list $lname [show_filetime $ltime] "n/a"]
			lappend transfer_list [list $cols $ltime]
		}
	}

	# any files to transfer?
	if {[llength $transfer_list] == 0} {
		show_msg . "There are no files that require updating."
		return
	}

	create_window rw "" "Remote Software Update: $setup(remote_database)"
	show_text rw "The following is a list of files to be updated.\nSelect each file to be updated or click on the \"Select All\" button for all." -relief groove

	set tw [select_list rw transfer_list "" "Files to Transfer" -cancel "" -multiple select_list -required 1 \
		-columns {{50 "Filename"} {20 "Local time"} {20 "Remote time"}} -height 25]

	input_button rw -actions {{"_Update" remote_update_now "" warning}} -actions [list [list "Select _All" "select_list_all $tw" novalidate start]] -buttons {{"Cancel" "" novalidate}}
	input_window rw
}

proc remote_update_now {} {
	global current_update
	variable transfer_list
	variable select_list
	variable transfer_note ""

	create_window trw rw "Transferring Files" -popup 1
	set tw [input_text trw transfer_note "" -expand yes -height 20 -width 80]
	input_window trw

	# transfer all selected files
	foreach index $select_list {
		set row [lindex $transfer_list $index]
		set cols [lindex $row 0]
		set file [lindex $cols 0]
		set mtime [lindex $row 1]

		set fid [open $file "r"]
		fconfigure $fid -translation binary
		set contents [read $fid]
		close $fid

		append transfer_note "Transferring file ... $file\n"
		system_widget_command $tw see end
		update

		remote_command "update_source" [list $file $mtime $contents]
	}

	# if system update is less than current update then automatically run "update.tcl" (via remote server)
	if {$login::system(LAST_SYSTEM_UPDATE) < $current_update} {
		append transfer_note "\nThe user must now press the 'Update System' button on the remote server.\n"
		$tw see end
		#update
		#if {[catch { remote_command "update_system" "" } msg]} {
		#	append transfer_note "update_system response: $msg"
		#}
	}

	input_button trw -buttons {{"_Close" "close_window rw" novalidate}}
	input_window trw
}

proc get_local_file_list {} {
	# get files, include standard files for system (exclude setup.dat)
	variable file_list {}

	add_files [pwd] "tclIndex"

	add_files [pwd] ".tcl"
	add_files [pwd]/images ".gif"
	add_files [pwd]/images ".jpg"
	add_files [pwd]/support ".dll"

	set new_list {}

	# now process all files and get the last modification time
	foreach file $file_list {
		set mtime [file mtime $file]
		lappend new_list [list $file $mtime]
	}

	return $new_list
}

# Append all matching files in dir to file_list
proc add_files { dir match } {
	variable file_list

	set pattern [file join $dir *]
	foreach fileName [lsort -dictionary [glob -nocomplain $pattern]] {
		if {[file isdirectory $fileName]} {
			add_files $fileName $match
			continue
		}
		if {![file isfile $fileName]} { continue }

		if {[string index $match 0] == "."} {
			# Extension
			if {[string tolower [file extension $fileName]] == $match} {
				add_file $fileName
			}
		} else {
			# Exact filename
			if {$match == [file tail $fileName]} {
				add_file $fileName
			}
		}
	}
}

# Append fileName to file_list, without xxx/new/
proc add_file { fileName } {
	variable file_list

	set f [string first "new/" $fileName]
	if {$f != -1} {
		set fileName [string range $fileName [expr {$f + 4}] end]
	}
	lappend file_list $fileName
}

proc show_filetime { time } {
	return [clock format $time -format "%d-%m-%y %I:%M%p"]
}

#--------------------------------------------------
# FBCLIENT - Initialise, connect etc.
#--------------------------------------------------

source common/environment.tcl
# don't call standard_libraries as the Fastbase software files may not be present, 
# i.e. when running standalone fbclient for the first time
# set standard globals
environment::standard_globals

# normally, the command arguments are evaluated at this point...

# first get the setup details from the configuration file, c:\fastbase.dat - this file must exist

global setup
# load configuration files, load setup.dat first (if exists)
if {[file exists setup.dat]} {
	source setup.dat
}

# command line options: -setup alternate_setup.dat_file
set options(-setup) ""

foreach {option value} $argv {
	set options($option) $value
}

if { $options(-setup) != "" } {
	source $options(-setup)
}

if { [file exists ../fastbase.dat] } {
	source ../fastbase.dat
}
# load the c:/fastbase.dat file also
if { [file exists c:/fastbase.dat] } {
	source c:/fastbase.dat
} elseif { [file exists /home/fastbase/fastbase.dat] } {
	source /home/fastbase/fastbase.dat
}

# if standalone set standard libraries to be not yet loaded (we're not calling statndard_libraries though)...

if { [setup fbproxy 1] } {
	# Proxy mode required for talking to a proxy-mode fbserver
	source common/proxy.tcl
	source common/fbproxy/softsql.tcl
} else {
	# Load the "fbsql.dll" extension for the "fbwait" command
	if { $tcl_platform(platform) == "unix" } {
		load "./fbsql.so"
	} else {
		load "./fbsql.dll"
	}
}

rename sql sql_unused

# sort the list of remote servers
set setup(remote_servers) [lsort -dictionary -index 0 $setup(remote_servers)]

remote_check_setup remote_port 5560
remote_check_setup remote_host "localhost"

remote_check_setup remote_timeout 300
remote_check_setup remote_polling 120

remote_check_setup remote_user ""
remote_check_setup remote_password ""
remote_check_setup remote_standalone 0

set setup(fbserver) 0

# determine the server to use
# provide main window for displaying server selection and/or file upload
wm title . "FastBase"

set server [select_remote_server]

# if no selected server then quit
if {![string is integer -strict $server]} { exit }

# assign the selected server details to the setup() array
set row [lindex $setup(remote_servers) $server]
set setup(remote_name) [lindex $row 0]
set setup(remote_host) [lindex $row 1]
set setup(remote_database) [lindex $row 2]
if {[lindex $row 3] != ""} {
	set setup(remote_port) [lindex $row 3]
}
if {[lindex $row 4] != ""} {
	set setup(remote_company_id) [lindex $row 4]
}

# display popup window with connection details
wm withdraw .
toplevel .splash -borderwidth 4 -relief raised
wm overrideredirect .splash 1

after idle {
	update idletasks

	# centre_window .splash
	if {[info commands .splash] == ""} { return }

	set xmax [winfo screenwidth .splash]
	set ymax [winfo screenheight .splash]
	set x [expr {($xmax - [winfo reqwidth .splash]) / 2}]
	set y [expr {($ymax - [winfo reqheight .splash]) / 2}]

	wm geometry .splash "+$x+$y"
}

label .splash.title -text "FastBase\nAccounting & Distribution System" -font {Arial 12 bold}
pack .splash.title -fill x -padx 8 -pady 6

label .splash.x1 -text "Connecting to server $setup(remote_host) on port $setup(remote_port) ..." -font {Arial 9}
label .splash.x2 -text "Remote User ID : $setup(remote_user)" -font {Arial 9}
pack .splash.x1 .splash.x2 -fill x -pady 2

# status
set setup(remote_file) ""
label .splash.status -font {Verdana 9 bold} -textvariable setup(remote_file) -justify left -anchor e -fg darkblue
pack .splash.status -side bottom -fill x

label .splash.info -text "FastBase is trying to connect to the remote database server $setup(remote_host) port $setup(remote_port).\nIf this message does not disappear then something may not be set-up correctly.\nHere is a small checklist;\n\t1. Make sure \"FastBase RA Server\" is running on the remote computer.\n\t2. Make sure the remote site's Internet firewall has been configured correctly, re-directing traffic on port $setup(remote_port) to the server.\n\t3. Check our web-site for further documentation." -font {Verdana 7} -justify left
label .splash.info2 -text "Documentation :: http://www.fastbase.co.nz" -font {Verdana 7}
pack .splash.info2 .splash.info -fill x -side bottom

update

# Connect to fbserver
source common/fbremote.tcl
if {[catch {fbremote::connect $setup(remote_host) $setup(remote_user) $setup(remote_password) \
							$setup(remote_database) $setup(remote_port)} msg]} {
	tk_messageBox -title "Connect Error" -icon error -type ok -message $msg
	exit
}

# now check the version of the system, only for stand-alone systems
if {$setup(remote_standalone)} {
	# check the user isn't running "fbclient" from a network drive! 15-05-2006
	# for the meantime, check the current directory doesn't start with anything other than "C:"
	set drive [string toupper [string range [pwd] 0 0]]
	if {$tcl_platform(platform) == "unix" || $drive == "C" || $drive == "D"} {
		get_new_version
	} else {
		error "Fbclient must be run from either the C:\ or D:\ drive when the \"remote_standalone\" flag is switched on.\nYour system is incorrectly configured, please contact your FastBase support staff for assistance."
	}

	# re-source the files we have already loaded, in case there have been changes:
	source support/combobox.tcl
	source common/command.tcl
	source widgets.tcl
	source toolkit/event.tcl
	initialise_widgets
}

# destroy popup window
destroy .splash

# now that we're connected rename our remote_sql command to "sql"
rename remote_sql sql

# connID used by query cache
set property(connID) $::fbremote::sockID

after [expr {$setup(remote_polling) * 1000}] { remote_polling }

# set database name (so we check correctly when running setup.dat script)
global argv
set argv [linsert $argv 0 $setup(remote_database)]

# now start fastbase programs
source as.tcl
