
-- Redis script to update object in redis (using hmset)
-- Publish the update to subscribers, publish any changes to zsets
-- Arguments:
--   domainTag     tag of database domain
--   objectTag     tag of object we're updating
--   sessionTag    tag of user session
--   instanceID    unique client-instance identifier
--   timestamp     time at client (unix time to usec)
--   {name value}  list of new/updated fields

local domainTag = KEYS[1]
local objectTag = KEYS[2]
local sessionTag = KEYS[3]
local instanceID = KEYS[4]
local timestamp = KEYS[5]

local objectKey = getObjectKey(domainTag, objectTag)
redis.log(redis.LOG_NOTICE, "update script " .. objectKey
				.. " session " .. sessionTag
				.. " instance " .. instanceID
				.. " timestamp " .. timestamp)

-- Get existing fields (as a dictionary)
local existingList = redis.call('hgetall', objectKey)
local existing = {}
addToSet(existing, existingList)

-- Get the objects existing types (not instances)
-- library ...
local objectIsKey = getTypeKey(domainTag, objectTag)
local existingTypeList = redis.call('zrangebyscore', objectIsKey, '-inf', -1, 'withscores')

local different = {}
local differentList = {}
local types = {}

-- Find fields that have changed including types
-- restructure ...
for i = 1, #ARGV, 2 do
	-- Each updated field
	local fieldTag = ARGV[i]
	local newValue = ARGV[i+1]
	local oldValue = existing[fieldTag] or ''

	redis.log(redis.LOG_NOTICE, fieldTag .. "=" .. newValue .. " (old=" .. oldValue .. ")" )

	if oldValue ~= newValue then
		different[fieldTag] = newValue
		table.insert(differentList, fieldTag)
		table.insert(differentList, newValue)

		if newValue == 'is' then
			-- A common ancestor will get the higher score ...
			local typeIsKey = getTypeKey(domainTag, fieldTag)
			local zrange = redis.call('zrangebyscore', typeIsKey, '-inf', 0, 'withscores')

			-- Ensure type is a type of itself
			if #zrange == 0 then
				types[fieldTag] = 0
				redis.log(redis.LOG_NOTICE, "types[" .. fieldTag .. "] = 0")
			else
				addToSet(types, zrange)
			end
		end
	end
end

-- All types (existing and changed)
local hasType = {}
for i = 1, #existingTypeList, 2 do
	local name = existingTypeList[i]
	local score = existingTypeList[i+1]
	hasType[name] = score
end

-- New or changed types
-- Convert from dictionary to score-member list (unsorted)
local typeList = {}
for name, score in pairs(types) do
	table.insert(typeList, score)
	table.insert(typeList, name)
	hasType[name] = score
end

redis.log(redis.LOG_NOTICE, "#typeList=" .. #typeList)

-- One or more types have changed
if #typeList > 0 then
	-- Get object instances (including itself)
	local instanceList = redis.call('zrangebyscore', objectIsKey, 0, '+inf', 'withscores')
	if #instanceList == 0 then
		table.insert(instanceList, objectTag)
		table.insert(instanceList, 0)
	end

	-- Merge the updated types into existing instances
	-- library ...
	local payload
	local previous = math.huge
	for i = 1, #instanceList, 2 do
		-- Each instance of object (including object itself)
		local name = instanceList[i]
		local score = instanceList[i+1]
		if score ~= previous then
			previous = score
			-- Decrement scores for types
			for i = 1, #typeList, 2 do
				typeList[i] = typeList[i]-1
			end

			-- Update payload to be published
			local object = {}
			for i = 1, #typeList, 2 do
				local score = typeList[i]
				local name = typeList[i+1]
				object[name] = score
			end
			local operation = {type='touch.type', instance=instanceID, fields=object}
			payload = cmsgpack.pack(operation)
		end

		-- Update instance with types
		local instanceIsKey = getTypeKey(domainTag, name)
		redisChunk('zadd', instanceIsKey, typeList)

		-- publish the change to object instance
		redis.call('publish', instanceIsKey, payload)
	end

	-- Get types sorted by score
	local sortedTypes = {}
	for name, score in pairs(types) do
		table.insert(sortedTypes, name)
	end
	table.sort(sortedTypes, function(a, b) return types[a] > types[b] end)

	-- create instanceArray (score-member)
	local instanceArray = {}
	for i = 1, #instanceList, 2 do
		local name = instanceList[i]
		local score = instanceList[i+1]
		table.insert(instanceArray, score)
		table.insert(instanceArray, name)
	end

	-- Merge object instances (including object itself) into types
	-- library ...
	previous = math.huge
	for i, name in ipairs(sortedTypes) do
		-- Each type (not including object itself)
		local score = types[name]
		if score ~= previous then
			previous = score
			-- Increment scores for descendants (the instances)
			for i = 1, #instanceArray, 2 do
				instanceArray[i] = instanceArray[i]+1
			end

			-- Update payload to be published
			local object = {}
			for i = 1, #instanceArray, 2 do
				local score = instanceArray[i]
				local name = instanceArray[i+1]
				object[name] = score
			end
			local operation = {type='touch.type', instance=instanceID, fields=object}
			payload = cmsgpack.pack(operation)
		end

		-- Update type with instances
		local typeIsKey = getTypeKey(domainTag, name)
		redisChunk('zadd', typeIsKey, instanceArray)

		-- publish the change to object type
		redis.call('publish', typeIsKey, payload)
	end

end -- one or more types changed

-- Update object and publish the changes
if #differentList > 0 then
	redisChunk('hmset', objectKey, differentList)

	local operation = {type='touch.object', instance=instanceID, fields=different}
	local payload = cmsgpack.pack(operation)
	redis.call('publish', objectKey, payload)

	-- Create "undo" object
	if hasType['tracked'] then
		-- Allocate undo object tag
		-- Or use timestamp, check if unique ...
		local editTag = redis.call('hincrby', ':' .. domainTag, 'tag', 1)
		local editKey = getObjectKey(domainTag, editTag)

		local editList = {'undo', 'is', 'object', objectTag}
		if sessionTag ~= '' then
			table.insert(editList, 'session')
			table.insert(editList, sessionTag)
		end
		if timestamp ~= '' then
			table.insert(editList, 'at')
			table.insert(editList, timestamp)
		end
		for name, newValue in pairs(different) do
			local oldValue = existing[name] or ''
			table.insert(editList, name)
			table.insert(editList, oldValue)
		end
		redisChunk('hmset', editKey, editList)

		if sessionTag ~= '' then
			-- Undo per session (delete, just local ...)
			local sessionUndoKey = domainTag .. ':undo.' .. sessionTag
			redis.call('rpush', sessionUndoKey, editTag)

			-- Clear redo stack per session (delete, just local ...)
			local sessionRedoKey = domainTag .. ':redo.' .. sessionTag
			redis.call('ltrim', sessionRedoKey, 1, 0)
		end

		-- Edits per object
		local objectEditKey = domainTag .. ':edit.' .. objectTag
		redis.call('rpush', objectEditKey, editTag)

		redis.log(redis.LOG_NOTICE, "undo " ..  editTag
					.. " object " .. objectTag)
	end
end

redis.log(redis.LOG_NOTICE, "update-script return")
-- Return edit tag ...
return {ok=''}
