
-- Redis script to undo an edit, reverse a previous change
-- Update object including types and instances, publish changes
-- Create an inverse edit (a redo edit)
-- Return tag of inverse edit (if an edit was created)
-- Arguments:
--   domainTag     tag of database domain
--   objectTag     tag of object, that edit applies to
--   sessionTag    tag of user session
--   instanceID    unique client-instance identifier
--   timestamp     time at client (unix time to usec)
--   editTag       tag of edit (edit to undo)

local domainTag = KEYS[1]
local objectTag = KEYS[2]
local sessionTag = KEYS[3]
local instanceID = KEYS[4]  -- not used ...
local timestamp = KEYS[5]
local editTag = KEYS[6]

-- Get existing fields (as a dictionary)
local objectKey = getObjectKey(domainTag, objectTag)
local existingList = redis.call('hgetall', objectKey)
local existing = {}
addToSet(existing, existingList)

-- Get the objects' existing types (not instances)
local objectIsKey = getTypeKey(domainTag, objectTag)
local objectIs = {}
addObjectAncestors(objectIs, objectIsKey)

-- Get the edit (that we're undoing)
local editKey = getObjectKey(domainTag, editTag)
local editList = redis.call('hgetall', editKey)
local edit = {}
addToSet(edit, editList)

-- Determine fields affected by undoing the previous edit
local differentList = {}
for name, undoValue in pairs(edit) do
	local redoValue = existing[name] or ''
	if undoValue ~= redoValue then
		table.insert(differentList, name)
		table.insert(differentList, undoValue)
	end
end

local inverseTag = ''

-- Undo edit to object
-- editList is the edit that we're undoing (empty = delete object)
if #editList == 0 or #differentList > 0 then
	local different = {}
	addToSet(different, differentList)

	-- Identify types, properties, parts, names; update zset indexes
	characteriseFields(domainTag, objectTag, different)

	-- Update or delete object fields
	for name, value in pairs(different) do
		if value ~= '' then
			redis.call('hset', objectKey, name, value)
		else
			redis.call('hdel', objectKey, name)
		end
	end

	-- Empty list of edit fields, delete the object
	if #editList == 0 then
		redis.call('del', objectKey)
	end

	-- Publish change to object (originator will not ignore)
	-- Send instanceID, have client not ignore ...
	local operation = {type='touch.object', instance='', fields=different}
	local payload = cmsgpack.pack(operation)
	redis.call('publish', objectKey, payload)

	-- Create inverse edit (that may itself be undone)
	if objectIs['tracked'] then
		-- Allocate tag for edit
		inverseTag = getUniqueTimestamp(domainTag, timestamp)
		local inverseKey = getObjectKey(domainTag, inverseTag)

		-- Create edit (only if object already existed)
		if #existingList > 0 then
			local inverseList = {}
			for name, undoValue in pairs(different) do
				local redoValue = existing[name] or ''
				table.insert(inverseList, name)
				table.insert(inverseList, redoValue)
			end
			redisChunk('hmset', inverseKey, inverseList)
		end

		-- Add edit to set of object edits
		local score = sessionTag
		if score == '' then score = 0 end
		local objectEditKey = getObjectEditKey(domainTag, objectTag)
		redis.call('zadd', objectEditKey, score, inverseTag)
	end
end

-- return tag of edit or ''
return {ok=inverseTag}


