
-- Redis script to undo change to a traced object
-- Apply change to object, the undo operation
-- Create redo object, append to object edit list
-- Update object type (also types and instances)
-- Publish the edit to subscribers, publish any changes to zsets
-- Arguments:
--   domainTag     tag of database domain
--   undoTag       tag of undo object
--   sessionTag    tag of user session
--   instanceID    unique client-instance identifier
--   timestamp     time at client (unix time to usec)

local domainTag = KEYS[1]
local undoTag = KEYS[2]
local sessionTag = KEYS[3]
local instanceID = KEYS[4]
local timestamp = KEYS[5]

local undoKey = getObjectKey(domainTag, undoTag)

redis.log(redis.LOG_NOTICE, "undo script " .. undoKey
				.. " session " .. sessionTag
				.. " instance " .. instanceID
				.. " timestamp " .. timestamp)

-- Get undo fields (as a dictionary)
local undoList = redis.call('hgetall', undoKey)
local undo = {}
addToSet(undo, undoList)
if not undo['undo'] then
	return {err='must be an undo object: ' .. undoKey}
end

local objectTag = undo['object']
local objectKey = getObjectKey(domainTag, objectTag)

-- List of edits to the object
local objectEditKey = domainTag .. ':edit.' .. objectTag

-- Get existing object fields (as a dictionary)
local existingList = redis.call('hgetall', objectKey)
local existing = {}
addToSet(existing, existingList)

-- Create different (fields to undo)
-- Check if actually different ...
-- Rename undoValue to editValue ...
-- Use '.' prefix on name ...
local different = {}
for name, undoValue in pairs(undo) do
	if name ~= 'undo' and name ~= 'object' and name ~= 'session' and name ~= 'at' then
		different[name] = undoValue
	end
end

-- Create redo
-- Could different be empty ...
local domainKey = getObjectKey('', domainTag)
local redoTag = redis.call('hincrby', domainKey, 'tag', 1)
local redoKey = getObjectKey(domainTag, redoTag)

local redoList = {'redo', 'is', 'object', objectTag}
if sessionTag ~= '' then
	table.insert(redoList, 'session')
	table.insert(redoList, sessionTag)
end
if timestamp ~= '' then
	table.insert(redoList, 'at')
	table.insert(redoList, timestamp)
end
for name, undoValue in pairs(different) do
	local redoValue = existing[name] or ''
	table.insert(redoList, name)
	table.insert(redoList, redoValue)
end
redisChunk('hmset', redoKey, redoList)
redis.call('rpush', objectEditKey, redoTag)

-- Apply undo to object
for name, undoValue in pairs(different) do
	if undoValue == '' then
		redis.call('hdel', objectKey, name)
	else
		redis.call('hset', objectKey, name, undoValue)
	end
end

-- Update the objects' types ...
-- Publish changes to types ...

-- Publish changes to object
local operation = {type='touch.object', instance=instanceID, fields=different}
local payload = cmsgpack.pack(operation)
redis.call('publish', objectKey, payload)

-- Return edit tag ...
return {ok=''}
