
# Connect to redis server and invoke it's services

# Result format
# Pipelining
# Detect connection closed, auto-reconnect
# Multiple connections

# ------------------------------------------------------------------------------------

namespace eval redis {

# Connection state (per sockID) - created on connect
# connected - 0/1
# lastError - socket error message
# instance - client instance identifier
# callback - proc for incoming messages
# level - multi-bulk stack level
# required - number of bulk-reply bytes to be read
# buffer - bulk-reply data already read
# elements (per level) - number of multi-bulk items to be received
# items (per level) - multi-bulk items already received
# requestID - last request sent to redis
# responseID - last response received from redis
# response (per responseID) - ready for receiver

# ---------------------------------------------------------------------------------

# Make a new connection to a redis server
# Return socket ID
proc connect { hostname {port ""} {incoming ""}} {
	variable connected
	variable lastError
	variable instance
	variable callback
	variable level
	variable required
	variable buffer
	variable elements
	variable items
	variable requestID
	variable responseID
	variable response
	variable err

	if {$port == ""} { set port 6379 }

	# Improve the socket error message (usually a DNS problem)
	if {[catch {set sock [socket -async $hostname $port]} msg]} {
		set reason ""
		regexp {^[^:]*: *(.*)$} $msg all reason
		if {$reason == "invalid argument"} { set reason "host is unreachable" }
		error "cannot connect to redis $hostname port $port:\n$reason"
	}

	# New connection state
	set connected($sock) 0
	set lastError($sock) ""
	set instance($sock) ""
	set callback($sock) $incoming

	set level($sock) 0
	set required($sock) 0
	set buffer($sock) ""
	array set elements {}
	array set items {}

	set requestID($sock) 0
	set responseID($sock) 0
	array set response {}
	array set err {}

	# Attempt connect
	fconfigure $sock -buffering full -translation binary -buffersize 65536 -blocking 0
	fileevent $sock writable [list ::redis::socket_writable $sock]
	vwait ::redis::connected($sock)
	if {! $connected($sock)} {
		error "cannot connect to redis $hostname port $port:\n$lastError($sock)"
	}

	# Connected
	fileevent $sock writable {}
	fileevent $sock readable [list ::redis::socket_readable $sock]
	return $sock
}

# Disconnect connection
proc disconnect { sock } {
	variable connected
	variable lastError
	variable instance
	variable callback
	variable requestID
	variable responseID
	variable response
	variable err

	if {$sock == ""} { return }
	if {! [info exists connected($sock)]} { return }
	if {! $connected($sock)} { return }

	catch { close $sock }
	set connected($sock) 0
	set instance($sock) ""
	set callback($sock) ""
	set requestID($sock) 0
	set responseID($sock) 0

	# Wakeup any receivers
	if {$lastError($sock) == ""} { set lastError($sock) "disconnected" }
	foreach index [array names response "$sock,*"] {
		set response($index) $lastError($sock)
		set err($index) 1
	}
}

# Connection socket is writable, check if connected OK
proc socket_writable { sock } {
	variable connected
	variable lastError

	set msg [fconfigure $sock -error]
	if {$msg != ""} {
		# Connect failed, e.g. "connection refused", "host is unreachable"
		set lastError($sock) $msg
		catch { close $sock }
		set connected($sock) 0
		return
	}
	set connected($sock) 1
}

# Read partial or complete response data from socket
# Also called when socket is closed by server
proc socket_readable { sock } {
	variable lastError
	variable level
	variable required
	variable buffer
	variable elements
	variable items

	if {$required($sock) == 0} {
		# Waiting for reply line
		set result [gets $sock line]
		if {$result < 0} {
			if {[eof $sock]} {
				set lastError($sock) "connection $sock has been closed"
				disconnect $sock
			}
			return
		}
		set first [string index $line 0]
		set content [string range $line 1 end-1]
		# puts "first $first content \"$content\""

		# + status reply - error reply : integer reply
		switch -exact -- $first {
			+ { }
			- { set_error $sock }
			: { }
			$ {
				if {$content == -1} {
					# NULL bulk reply
					set required($sock) 0
					set content ""
				} else {
					# Bulk reply (length plus cr lf)
					set required($sock) [expr {$content + 2}]
				}
			}
			* {
				if {$content > 0} {
					# Start of multi-bulk reply
					incr level($sock)
					set index "$sock,$level($sock)"
					set elements($index) $content
					set items($index) {}
					return
				}
				# Empty or NULL multi-bulk reply
				set content {}
			}
			default {
				set lastError($sock) "bad redis protocol reply type"
				disconnect $sock
				return
			}
		}
	}

	# EOF after reply line
	if {[eof $sock]} {
		set lastError($sock) "truncated reply from redis server"
		disconnect $sock
		return
	}

	# Expecting bulk data
	if {$required($sock) > 0} {
		set data [read $sock $required($sock)]
		set received [string length $data]

		append buffer($sock) $data
		set required($sock) [expr {$required($sock) - $received}]
		# Didn't read all of it
		if {$required($sock) > 0} { return }

		# Got all the bulk data, ignore cr lf
		set content [string range $buffer($sock) 0 end-2]
		set buffer($sock) ""
	}

	while {true} {
		# Not in multi-block, return content
		if {$level($sock) == 0} {
			set_response $sock $content
			break
		}
		# Within multi-block, append
		set index "$sock,$level($sock)"
		lappend items($index) $content
		incr elements($index) -1
		if {$elements($index)} { break }

		# At end of multi-block, finished building item list
		set content $items($index)
		set items($index) {}
		incr level($sock) -1
	}
}

# Received an error reply, flag error for receiver
proc set_error { sock } {
	variable responseID
	variable err

	set id $responseID($sock)
	incr id
	set index "$sock,$id"
	set err($index) 1
}

# Received complete reply message, set response (for receiver)
proc set_response { sock result } {
	variable instance
	variable callback
	variable responseID
	variable response
	variable err

	set id [incr responseID($sock)]
	set index "$sock,$id"

	# Async connection (usually publish/subscribe)
	if {$callback($sock) != ""} {
		set errorResult 0
		if {[info exists err($index)]} {
			unset err($index)
			set errorResult 1
		}
		# temp ...
		set result [$callback($sock) $instance($sock) $result $errorResult]
		puts $result
		return
	}

	if {[info exists response($index)]} {
		# No receiver waiting, discard reply
		unset response($index)
		catch { unset err($index) }
	} else {
		# Wakeup receiver (waiting on vwait)
		set response($index) $result
	}
}

# Encode argList as a redis request message, write to socket
# If discard is true, discard received response
# Return request ID
proc transmit { sock argList {discard 0}} {
	variable connected
	variable requestID
	variable response
	variable lastError

	if {! $connected($sock)} {
		if {$lastError($sock) != ""} {
			error "cannot send redis request: $lastError($sock)"
		}
		error "redis::transmit: not connected"
	}

	# Allocate ID
	set id [incr requestID($sock)]
	set index "$sock,$id"

	# Discard response when it arrives
	if {$discard} {
		set response($index) ""
	}

	# Encode request message
	set count [llength $argList]
	set request "*${count}\r\n"
	foreach arg $argList {
		set length [string length $arg]
		append request "\$${length}\r\n${arg}\r\n"
	}
	puts -nonewline $sock $request
	flush $sock
	return $id
}

# Receive the response for this request ID, wait if necessary
# Return received data
proc receive { sock id } {
	variable connected
	variable lastError
	variable response
	variable err

	# Wait for response (set by socket_readable/set_response)
	set index "$sock,$id"
	while {! [info exists response($index)]} {
		if {! $connected($sock)} {
			error "redis::receive: not connected"
		}
		vwait ::redis::response($index)
	}
	set result $response($index)
	unset response($index)

	# Throw error in caller
	if {[info exists err($index)]} {
		unset err($index)
		return -code error $result
	}
	return $result
}

# Set instance identifier for a connection
proc set_instance { sock instanceID } {
	variable instance

	set instance($sock) $instanceID
}

# Get instance identifier for a connection
proc get_instance { sock } {
	variable instance

	return $instance($sock)
}

}
