
# Generic redis objects and their types, edit/undo/redo

# ------------------------------------------------------------------------------------
namespace eval redis {

	# Undo and redo state (per instance/session)
	variable undoStack {}
	variable redoStack {}
	# Object per edit
	array set editObject {}

# ---------------------------------------------------------------------------------

# Return redis key for a domain object
proc objectkey { domainTag objectTag } {
	return "${domainTag}:${objectTag}"
}

# Return redis key for a group (ie. type, whole or place)
proc groupkey { domainTag index groupTag } {
	return "${domainTag}:${index}.${groupTag}"
}

# Return value of object's field (or empty string if field doesn't exist)
# Delete this proc ...
# still used ? ...
proc field { objectVar name } {
	if {[catch { set value [set ${objectVar}($name)] }]} { return "" }
	return $value
}

# ---------------------------------------------------------------------------------

# Create initial objects for the domain
# Per trading-world-0.1
proc create_domain_objects { domainTag } {
	set property(domain) $domainTag

	# Domain
	set userVar [redis::bind_object "user"]
	set ${userVar}(is.person) 1
	set ${userVar}(is.tracked) 1
	redis::unbind_object "user"

	set editVar [redis::bind_object "edit"]
	set ${editVar}(is.event) 1
	redis::unbind_object "edit"

	set noteVar [redis::bind_object "note"]
	set ${noteVar}(is.document) 1
	redis::unbind_object "note"

	set deviceVar [redis::bind_object "device"]
	set ${deviceVar}(is.tracked) 1
	redis::unbind_object "device"

	set printerVar [redis::bind_object "printer"]
	set ${printerVar}(is.tracked) 1
	redis::unbind_object "printer"

	set documentVar [redis::bind_object "document"]
	set ${documentVar}(is.tracked) 1
	redis::unbind_object "document"

	set layoutVar [redis::bind_object "layout"]
	set ${layoutVar}(is.document) 1
	redis::unbind_object "layout"

	set ticketVar [redis::bind_object "ticket"]
	set ${ticketVar}(is.document) 1
	redis::unbind_object "ticket"

	# Entity
	set entityVar [redis::bind_object "entity"]
	set ${entityVar}(is.tracked) 1
	redis::unbind_object "entity"

	set personVar [redis::bind_object "person"]
	set ${personVar}(is.entity) 1
	set ${personVar}(is.contact) 1
	redis::unbind_object "person"

	set companyVar [redis::bind_object "company"]
	set ${companyVar}(is.entity) 1
	redis::unbind_object "company"

	# Role
	set roleVar [redis::bind_object "role"]
	set ${roleVar}(is.abstract) 1
	set ${roleVar}(is.tracked) 1
	redis::unbind_object "role"

	set othersVar [redis::bind_object "others"]
	set ${othersVar}(is.role) 1
	set ${othersVar}(is.balancesheet) 1
	redis::unbind_object "others"

	set supplierVar [redis::bind_object "supplier"]
	set ${supplierVar}(is.role) 1
	set ${supplierVar}(is.tradeaccount) 1
	redis::unbind_object "supplier"

	set customerVar [redis::bind_object "customer"]
	set ${customerVar}(is.role) 1
	set ${customerVar}(is.tradeaccount) 1
	redis::unbind_object "customer"

	set carrierVar [redis::bind_object "carrier"]
	set ${carrierVar}(is.role) 1
	set ${carrierVar}(is.carrieraccount) 1
	set ${carrierVar}(is.supplier) 1
	redis::unbind_object "carrier"

	set cashierVar [redis::bind_object "cashier"]
	set ${cashierVar}(is.role) 1
	set ${cashierVar}(is.cashaccount) 1
	redis::unbind_object "cashier"

	set branchVar [redis::bind_object "branch"]
	set ${branchVar}(is.role) 1
	redis::unbind_object "branch"

	set bankVar [redis::bind_object "bank"]
	set ${bankVar}(is.role) 1
	set ${bankVar}(is.bankaccount) 1
	redis::unbind_object "bank"

	# Contacts and places
	set addressVar [redis::bind_object "address"]
	set ${addressVar}(is.tracked) 1
	redis::unbind_object "address"

	set contactVar [redis::bind_object "contact"]
	set ${contactVar}(is.role) 1
	redis::unbind_object "contact"

	set countryVar [redis::bind_object "country"]
	set ${countryVar}(is.tracked) 1
	redis::unbind_object "country"

	set locationVar [redis::bind_object "location"]
	set ${locationVar}(is.tracked) 1
	redis::unbind_object "location"

	# Structure of things
	set thingVar [redis::bind_object "thing"]
	set ${thingVar}(is.tracked) 1
	redis::unbind_object "thing"

	set substanceVar [redis::bind_object "substance"]
	set ${substanceVar}(is.thing) 1
	redis::unbind_object "substance"

	set unitVar [redis::bind_object "unit"]
	set ${unitVar}(is.thing) 1
	set ${unitVar}(is.bounded) 1
	redis::unbind_object "unit"

	set itemVar [redis::bind_object "item"]
	set ${itemVar}(is.thing) 1
	set ${itemVar}(is.bounded) 1
	redis::unbind_object "item"

	set quantityVar [redis::bind_object "quantity"]
	set ${quantityVar}(is.thing) 1
	set ${quantityVar}(is.bounded) 1
	redis::unbind_object "quantity"

	set valueVar [redis::bind_object "value"]
	set ${valueVar}(is.thing) 1
	set ${valueVar}(is.quantity) 1
	redis::unbind_object "value"

	set countVar [redis::bind_object "count"]
	set ${countVar}(is.thing) 1
	set ${countVar}(is.quantity) 1
	redis::unbind_object "count"

	set wholeVar [redis::bind_object "whole"]
	set ${wholeVar}(is.thing) 1
	set ${wholeVar}(is.bounded) 1
	redis::unbind_object "whole"

	set partVar [redis::bind_object "part"]
	set ${partVar}(is.thing) 1
	set ${partVar}(is.bounded) 1
	redis::unbind_object "part"

	# Properties of things
	set nameVar [redis::bind_object "name"]
	set ${nameVar}(is.property) 1
	redis::unbind_object "name"

	set descriptionVar [redis::bind_object "description"]
	set ${descriptionVar}(is.name) 1
	redis::unbind_object "description"

	set alternateVar [redis::bind_object "alternate"]
	set ${alternateVar}(is.property) 1
	redis::unbind_object "alternate"

	set supersedesVar [redis::bind_object "supersedes"]
	set ${supersedesVar}(is.property) 1
	redis::unbind_object "supersedes"

	# Nature of things
	set mobileVar [redis::bind_object "mobile"]
	set ${mobileVar}(is.physical) 1
	redis::unbind_object "mobile"

	set transientVar [redis::bind_object "transient"]
	set ${transientVar}(is.abstract) 1
	redis::unbind_object "transient"

	set moneyVar [redis::bind_object "money"]
	set ${moneyVar}(is.abstract) 1
	set ${moneyVar}(is.enduring) 1
	redis::unbind_object "money"

	set currencyVar [redis::bind_object "currency"]
	set ${currencyVar}(is.money) 1
	set ${currencyVar}(is.substance) 1
	set ${currencyVar}(is.unit) 1
	redis::unbind_object "currency"

	set holdingVar [redis::bind_object "holding"]
	set ${holdingVar}(is.abstract) 1
	set ${holdingVar}(is.enduring) 1
	redis::unbind_object "holding"

	set rightVar [redis::bind_object "right"]
	set ${rightVar}(is.abstract) 1
	set ${rightVar}(is.enduring) 1
	redis::unbind_object "right"

	# Priced thing
	set pricedthingVar [redis::bind_object "pricedthing"]
	set ${pricedthingVar}(is.thing) 1
	set ${pricedthingVar}(is.bounded) 1
	set ${pricedthingVar}(is.price) "has"
	redis::unbind_object "pricedthing"

	set pricedquantityVar [redis::bind_object "pricedquantity"]
	set ${pricedquantityVar}(is.thing) 1
	set ${pricedquantityVar}(is.quantity) 1
	set ${pricedquantityVar}(is.price) "has"
	redis::unbind_object "pricedquantity"

	# Account
	set accountVar [redis::bind_object "account"]
	set ${accountVar}(is.role) 1
	set ${accountVar}(is.abstract) 1
	set ${accountVar}(is.tracked) 1
	redis::unbind_object "account"

	set balancesheetVar [redis::bind_object "balancesheet"]
	set ${balancesheetVar}(is.account) 1
	redis::unbind_object "balancesheet"

	set cashaccountVar [redis::bind_object "cashaccount"]
	set ${cashaccountVar}(is.account) 1
	redis::unbind_object "cashaccount"

	set bankaccountVar [redis::bind_object "bankaccount"]
	set ${bankaccountVar}(is.account) 1
	redis::unbind_object "bankaccount"

	set tradeaccountVar [redis::bind_object "tradeaccount"]
	set ${tradeaccountVar}(is.account) 1
	redis::unbind_object "tradeaccount"


	# set eventVar [redis::bind_object "event"]
	# set ${eventVar}(is.object) 1
	# redis::unbind_object "event"

}

# ---------------------------------------------------------------------------------
# bind_object - bind to object
# bind_type - bind to type/instance list
# bind_whole - bind to whole/part list
# bind_place - bind to place/point list

# undo_edit - undo a previous edit
# redo_edit - redo a previous undo

# shallow or deep ...
# bind to list of objects (streaming) ...
# auto-generate tag (or timestamp) ...
# In as.tcl bind to domain object using -domain ""

# Bind to an object, get current value of object fields
# The object may have zero fields (ie. doesn't exist)
# Arrange for changes to be received, and local changes to be sent
# Bind to an object that doesn't exist will create object
# Return name of object array variable
# options
# domain - tag of domain containing the object
# object - tag of object to be bound
# var - full name of object array variable, default "::domain.<domain>::<object>" (deprecated...)
# callback - proc to call on incoming change
proc bind_object { args } {
	set defaults {var ""}
	command::arguments $args opt {object} $defaults
	variable bound
	variable dirty
	variable where

	if {[info exists opt(domain)]} {
		set domainTag $opt(domain)
		set property(domain) $domainTag
	} else {
		set domainTag [command::inherit property {domain}]
	}
	set objectTag $opt(object)
	set objectVar $opt(var)

	# logToFile "bind_object domainTag $domainTag objectTag $objectTag"

	# Variable storing the object, default "::domain.<domain>::<object>"
	if {$objectVar == ""} {
		set domain_ns "::domain.${domainTag}"
		set objectVar "${domain_ns}::${objectTag}"
		namespace eval $domain_ns {}
	}

	# Are we already bound to the object
	set objectIndex "${domainTag}|${objectTag}"
	if {[info exists bound($objectIndex)]} {
		if {$objectVar != $where($objectIndex)} {
			error "bind_object: already bound to different variable"
		}
		incr bound($objectIndex)
		return $objectVar
	}

	# Not already bound
	set where($objectIndex) $objectVar
	set key [redis::objectkey $domainTag $objectTag]
	set bound($objectIndex) 1

	# Subscribe to channel (object key)
	redis::subscribe $key

	# Get current object value
	set fields [redis hgetall $key]
	array set $objectVar $fields

	# At this point we are in sync with redis
	# Dirty list usually does not exist
	enable_trace $objectVar $objectIndex
	return $objectVar
}

# Bind to a type, get current value of types/instances
# Return name of type/instance array variable
# options
# domain - tag of domain containing the type
# type - tag of type to be bound
proc bind_type { args } {
	return [eval "bind_group -class is $args"]
}

# Bind to a whole, get current value of whole/parts
# Return name of whole/part array variable
# options
# domain - tag of domain containing the whole/part
# whole - tag of whole-object to be bound
proc bind_whole { args } {
	return [eval "bind_group -class of $args"]
}

# Bind to a place, get current value of place/points
# Return name of place/point array variable
# options
# domain - tag of domain containing the place/point
# place - tag of place-object to be bound
proc bind_place { args } {
	return [eval "bind_group -class in $args"]
}

# Bind to group (type, whole or place) - internal proc
# Arrange for changes to be received (local changes cannot be sent)
# Return name of array variable
proc bind_group { args } {
	set defaults {}
	command::arguments $args opt {group} $defaults
	variable bound
	variable where

	if {[info exists opt(domain)]} {
		set domainTag $opt(domain)
		set property(domain) $domainTag
	} else {
		set domainTag [command::inherit property {domain}]
	}

	# Class of group, ie. "is" "of" "in"
	set class $opt(class)
	set groupTag $opt(group)

	# Variable storing the group members
	set domain_ns "::domain.${domainTag}"
	set groupVar "${domain_ns}::${class}.${groupTag}"
	namespace eval $domain_ns {}

	# Are we already bound to the type
	set groupIndex "${domainTag}|${class}.${groupTag}"
	if {[info exists bound($groupIndex)]} {
		incr bound($groupIndex)
		return $groupVar
	}

	# Not already bound
	set where($groupIndex) $groupVar
	set key [redis::groupkey $domainTag $class $groupTag]
	set bound($groupIndex) 1

	# Subscribe to channel (group key)
	# Don't need to trace (no local changes)
	redis::subscribe $key

	# Get current group members
	set scores [redis zrangebyscore $key "-inf" "+inf" "withscores"]
	array set $groupVar $scores
	return $groupVar
}

# Enable trace on object
# Trace when elements of the array are written, or array is deleted
proc enable_trace { objectVar objectIndex } {
	set cmd [list redis::trace_write $objectIndex]
	trace add variable $objectVar write $cmd
}

# Disable trace on object
proc disable_trace { objectVar objectIndex } {
	set cmd [list redis::trace_write $objectIndex]
	trace remove variable $objectVar write $cmd
}

# Called upon write to a traced array element
# Also called if array has been deleted (name2 will be empty)
# The dirty list may not exist
proc trace_write { objectIndex name1 name2 op } {
	lappend ::redis::dirty($objectIndex) $name2
}

# Unbind from the object, destroy object array (if top-level bind)
# No error if not already bound
# Different bind levels could use different array variables ...
# options
# domain - tag of domain containing the object
# object - tag of object to be unbound
proc unbind_object { args } {
	set defaults {}
	command::arguments $args opt {object} $defaults
	variable bound
	variable dirty
	variable where

	if {[info exists opt(domain)]} {
		set domainTag $opt(domain)
		set property(domain) $domainTag
	} else {
		set domainTag [command::inherit property {domain}]
	}
	set objectTag $opt(object)
	set objectIndex "${domainTag}|${objectTag}"

	# Are we bound to the object
	if {! [info exists bound($objectIndex)]} { return }
	if {$bound($objectIndex) > 1} {
		incr bound($objectIndex) -1
		return
	}

	set objectVar $where($objectIndex)
	if {$bound($objectIndex) == 1} {
		disable_trace $objectVar $objectIndex
		flush_object $objectTag
		set key [redis::objectkey $domainTag $objectTag]
		unsubscribe $key
		set bound($objectIndex) 0
	}
	# Destroy object array etc
	catch { array unset $objectVar }
	catch { unset dirty($objectIndex) }
	unset where($objectIndex)
	unset bound($objectIndex)
}

# Unbind from the type, destroy type array (if top-level bind)
# options
# domain - tag of domain containing the type
# type - tag of type to be unbound
# Fix like bind_index ... 
proc unbind_type { args } {
	set defaults {}
	command::arguments $args opt {type} $defaults
	variable bound
	variable where

	if {[info exists opt(domain)]} {
		set domainTag $opt(domain)
		set property(domain) $domainTag
	} else {
		set domainTag [command::inherit property {domain}]
	}

	set class xxx

	set groupTag $opt(type)
	set groupIndex "${domainTag}|${class}.${groupTag}"

	# Are we bound to the type
	if {! [info exists bound($groupIndex)]} { return }
	if {$bound($groupIndex) > 1} {
		incr bound($groupIndex) -1
		return
	}
	set groupVar $where($groupIndex)
	if {$bound($groupIndex) == 1} {
		set key [redis::groupkey $domainTag $class $groupTag]
		unsubscribe $key
		set bound($groupIndex) 0
	}
	# Destroy type array etc
	catch { array unset $groupVar }
	unset where($groupIndex)
	unset bound($groupIndex)
}

# Flush all dirty changes to server (for all domains)
# Called after idle, and upon timeout
# Also control reconnection ...
proc flush_all { redisID } {
	variable dirty
	variable after_idle
	variable after_id

	# Just gone idle, delay a little
	if {$after_idle != ""} {
		set after_idle ""
		if {$after_id != ""} { after cancel $after_id }
		set after_id [after 5000 ::redis::flush_all $redisID]
		return
	}
	set after_id ""

	# Each dirty object
	set property(redisID) $redisID
	foreach {name value} [array get dirty] {
		flush_objectIndex $redisID $name
	}

	# Re-enable flush upon idle
	set after_idle [after idle ::redis::flush_all $redisID]
}

# Flush object, objectIndex is "domain|object"
proc flush_objectIndex { redisID objectIndex } {
	# Get domain and object tags
	set pipe [string first "|" $objectIndex]
	if {$pipe == -1} { return }
	set domainTag [string range $objectIndex 0 [expr {$pipe -1}]]
	set objectTag [string range $objectIndex [expr {$pipe +1}] end]

	set property(domain) $domainTag
	flush_dirtyObject $redisID $domainTag $objectTag
}

# Flush an object (public method)
proc flush_object { objectTag } {
	variable bound
	variable dirty

	set redisID [command::inherit property {redisID}]
	set domainTag [command::inherit property {domain}]
	set objectIndex "${domainTag}|${objectTag}"

	# Is object bound and dirty (needs flushing)
	if {! [info exists bound($objectIndex)]} { return }
	if {$bound($objectIndex) < 1} { return }
	if {! [info exists dirty($objectIndex)]} { return	}
	if {[llength $dirty($objectIndex)] == 0} { return	}

	flush_dirtyObject $redisID $domainTag $objectTag
}

# Flush dirty changes on object to server (internal method)
# Assumes object is bound
# Compare current with original value, only flush if changed ..
proc flush_dirtyObject { redisID domainTag objectTag } {
	variable dirty
	variable where
	variable undoStack
	variable redoStack
	variable editObject

	# Get current user session (optional)
	set sessionTag ""
	catch { set sessionTag [command::inherit property {session}] }

	set objectIndex "${domainTag}|${objectTag}"
	if {! [info exists dirty($objectIndex)]} { return }

	# Get values of dirty fields
	# Per which list, if multiple bound variables ...
	set objectVar $where($objectIndex)
	array set object {}
	foreach field $dirty($objectIndex) {
		set object($field) [set ${objectVar}($field)]
	}
	# Clear dirty list
	unset dirty($objectIndex)

	# Script keys and arguments
	set instanceID [redis::get_instance $redisID]
	set timestamp [redis::timestamp]
	set keyList [list $domainTag $objectTag $sessionTag $instanceID $timestamp]
	set argList {}
	foreach {name value} [array get object] {
		lappend argList $name
		lappend argList $value
	}
	set editTag [redis::run_script {"library.lua" "edit.lua"} $keyList $argList]
	
	# Save edit (for undo)
	if {$editTag != ""} {
		lappend undoStack $editTag
		set editObject($editTag) $objectTag 
		set redoStack {} 
	}
}

# Undo the previous edit (if any)
proc undo_edit { redisID } {
	variable undoStack
	variable redoStack
	variable editObject
	
	set length [llength $undoStack]
	if {$length == 0} { return}
	set index [expr {$length -1}]
	set editTag [lindex $undoStack $index]
	
	# Get current user session (optional)
	set sessionTag ""
	catch { set sessionTag [command::inherit property {session}] }
	
	# Script keys and arguments
	set domainTag [command::inherit property {domain}]
	set instanceID [redis::get_instance $redisID]
	set timestamp [redis::timestamp]
	set objectTag $editObject($editTag)

	set keyList [list $domainTag $objectTag $sessionTag $instanceID $timestamp $editTag]
	set argList {}
	set inverseTag [run_script {"library.lua" "undo.lua"} $keyList $argList]
	
	# Pop undo from stack
	incr index -1
	set undoStack [lrange $undoStack 0 $index]
	
	if {$inverseTag != ""} {
		# Push inverse edit (redo) on redo stack
		lappend redoStack $inverseTag
		set editObject($inverseTag) $objectTag 
	}
}

# Redo the previous undo
proc redo_edit { } {
	variable undoStack
	variable redoStack
	variable editObject
		
	set length [llength $redoStack]
	if {$length == 0} { return}
	set index [expr {$length -1}]
	set editTag [lindex $redoStack $index]
	
	# Get current user session (optional)
	set sessionTag ""
	catch { set sessionTag [command::inherit property {session}] }
	
	# Script keys and arguments
	set domainTag [command::inherit property {domain}]
	set instanceID [redis::get_instance $redisID]
	set timestamp [redis::timestamp]
	set objectTag $editObject($editTag)

	set keyList [list $domainTag $objectTag $sessionTag $instanceID $timestamp $editTag]
	set argList {}
	set inverseTag [run_script {"library.lua" "undo.lua"} $keyList $argList]
	
	# Pop redo from stack
	incr index -1
	set redoStack [lrange $redoStack 0 $index]
	
	if {$inverseTag != ""} {
		# Push inverse edit (undo) on undo stack
		lappend undoStack $inverseTag
		set editObject($inverseTag) $objectTag 
	}
}

}
