
# Redis connection management, client instance

# -----------------------------------------------------------------------
namespace eval redis {

	array set subscribed {}

	variable last_server_time ""
	variable last_client_time ""

	# Add script_mtime, script_use ...
	array set script_sha1 {}
	array set script_body {}

	variable after_idle ""
	variable after_id ""

# -----------------------------------------------------------------------

# Create new client instance
# Create main redis connection and subscribe connection
# Called on startup and on reconnection ...
# Currently storing redisID and subscribeID in property ...
# But if connection is closed (externally) these IDs become invalid ...
# Should store a single instance ref, with the two IDs and a "connected" flag ...
# For replication, a masterID and slaveID, subscribeID is to slave only ...
# incoming "self" object edit/undo tells us the slave is in sync ...
proc new_instance { propertyName server {port ""} } {
	upvar $propertyName property
	variable subscribed
	variable after_idle
	variable after_id

	# Main connection
	set redisID [redis::connect $server $port]
	set property(redisID) $redisID

	# Subscribe connection
	set subscribeID [redis::connect $server $port redis::incoming]
	set property(subscribeID) $subscribeID

	# Allocate instanceID, set on both connections
	# InstanceID needs to be unique across all clients
	set instanceID [redis::server_time]
	redis::set_instance $redisID $instanceID
	redis::set_instance $subscribeID $instanceID

	# Re-subscribe (if we're reconnecting)
	# Also re-get current value ...
	foreach {name value} [array get subscribed] {
		redis::subscribe $name
	}

	# Enable flush after (idle plus delay)
	# Flush behaviour for non-gui (no event loop)...
	# Should use instanceID as after key ...
	# Lookup current redisID (and subscribeID) from instance ...

	if {$after_idle == ""} {
		set after_idle [after idle ::redis::flush_all $redisID]
	}
	return $instanceID
}

# -----------------------------------------------------------------------
# Time

# Get timestamp, unix time to usecs
# Use local client time adjusted to redis server time
proc timestamp { } {
	variable last_server_time
	variable last_client_time

	if {$last_server_time != ""} {
		# Find duration since we last got server time
		set client_time [redis::client_time]
		set duration [expr {$client_time - $last_client_time}]
		if {$duration >= 0.0 && $duration < 900.0} {
			set timestamp [format %.6f [expr {$last_server_time + $duration}]]
		} else {
			set last_server_time ""
		}
	}
	if {$last_server_time == ""} {
		# Get server time
		set begin [redis::client_time]
		set last_server_time [redis::server_time]
		set end [redis::client_time]
		set roundTrip [expr {$end - $begin}]

		# Transit = half the total roundtrip delay
		set transit 0.0
		if {$roundTrip > 0.0} { set transit [expr {$roundTrip / 2.0}] }
		set last_client_time [expr {$begin + $transit}]
		set timestamp [format %.6f [expr {$last_server_time + $transit}]]
	}
	return $timestamp
}

# Get redis server time
proc server_time { } {
	set reply [redis "time"]
	set sec [lindex $reply 0]
	set usec [lindex $reply 1]
	return "${sec}.[format %.6d $usec]"
}

# Get local client time, unit seconds, resolution usec
# For duration only: epoch is arbitrary, may sometimes wrap
proc client_time { } {
	global tcl_version

	if {$tcl_version >= 8.5} {
		set usecs [clock microseconds]
		return [format %.6f [expr {$usecs / 1000000.0}]]
	}
	set msecs [clock clicks -milliseconds]
	return [format %.6f [expr {$msecs / 1000.0}]]
}

# Allocate a new event tag using server time
proc new_event_tag { } {
	return [redis::timestamp]
}

# Allocate a new session tag
proc new_session_tag { } {
	return [redis::timestamp]
}

# -----------------------------------------------------------------------
# Lua scripts

# Read body of lua script from file(s)
proc read_script { scriptList } {
	set script ""
	foreach scriptFile $scriptList {
		set pathname [file join "common/redis" $scriptFile]
		set fid [open $pathname]
		set data [read $fid]
		close $fid

		append script $data
	}
	return $script
}

# Determine sha1 hash of lua script
proc get_script_hash { script } {
	if {[catch { package require sha1 } msg]} {
		# Can't calculate hash locally, use redis
		set sha1 [redis script load $script]
	} else {
		# Calculate hash locally
		set sha1 [sha1::sha1 $script]
	}
	return $sha1
}

# Load lua script from files (if necessary)
# Execute lua script on redis server
proc run_script { scriptList keyList argList } {
	variable script_body
	variable script_sha1

	if {[info exists script_body($scriptList)]} {
		# Script files have already been read
		set script $script_body($scriptList)
		set sha1 $script_sha1($scriptList)
	} else {
		# Read script from files
		set script [read_script $scriptList]
		set sha1 [get_script_hash $script]

		set script_body($scriptList) $script
		set script_sha1($scriptList) $sha1
	}

	set numKeys [llength $keyList]
	set evalList [concat [list "evalsha" $sha1 $numKeys] $keyList $argList]

	if {[catch { set result [redis_command $evalList] } msg]} {
		if {! [string match -nocase "*NOSCRIPT*" $msg]} {
			# Different error
			error $msg $::errorInfo
		}
		# Script isn't loaded, load then try again
		redis script load $script
		set result [redis_command $evalList]
	}

	# Re-start timer to expire script ...
	return $result
}

# -----------------------------------------------------------------------
# Publish/subscribe

# Subscribe (via separate redis connection)
# Channel is usually "domain:object"
# Return false if already subscribed
proc subscribe { channel } {
	variable subscribed
	set subscribeID [command::inherit property {subscribeID}]

	if {[info exists subscribed($channel)]} { return 0 }
	set discard 1
	set argList [list "subscribe" $channel]
	set id [redis::transmit $subscribeID $argList $discard]
	
	puts "subscribe $channel"

	# Remember subscription (for reconnect)
	set subscribed($channel) 1
	return 1
}

# Unsubscribe (via separate redis connection)
# Return false if not subscribed
proc unsubscribe { channel } {
	variable subscribed
	set subscribeID [command::inherit property {subscribeID}]

	if {! [info exists subscribed($channel)]} { return 0 }
	set discard 1
	set argList [list "unsubscribe" $channel]
	set id [redis::transmit $subscribeID $argList $discard]

	puts "unsubscribe $channel"
	
	# Remove subscription
	unset subscribed($channel)
}

# Received incoming async message (usually from publish)
proc incoming { instanceID result errorResult} {
	variable bound
	variable dirty
	variable where

	set debug "incoming: instanceID '$instanceID'"

	if {$errorResult} {
		append debug " errorResult $result"
		return $debug
	}
	# Ignore response from subscribe and unsubscribe
	set kind [lindex $result 0]
	if {$kind != "message"} { append debug " (kind='$kind')"; return $debug }

	# Published message channel id and content
	set channel [lindex $result 1]
	set payload [lindex $result 2]

	# Parse channel (object key) get domain and object tags
	# For a type, objectTag will be "is.<typeTag>"
	# set property(domain) on the stack if we invoke anything ... 
	set colon [string first ":" $channel]
	if {$colon == -1} { error "redis::incoming invalid channel '$channel'" }
	set domainTag [string range $channel 0 [expr {$colon -1}]]
	set objectTag [string range $channel [expr {$colon +1}] end]

	# Decode message (encoded in lua)
	set decoded [msgpack::unpack $payload]
	array set msg $decoded

	# Did we publish the message
	set sender $msg(instance)
	append debug " decoded='$decoded' sender='$sender'"

	if {$instanceID == $sender} {
		append debug " self"
		return $debug
	}
	append debug " other"

	# Ignore message if not bound to the object
	set objectIndex "${domainTag}|${objectTag}"
	if {! [info exists bound($objectIndex)]} {
		append debug " (not bound)"
		return $debug
	}
	if {$bound($objectIndex) == 0} {
		append debug " (not bound)"
		return $debug
	}

	append debug " (bound to '$objectIndex' type='$msg(type)')"

	# Current object fields
	set objectVar $where($objectIndex)

	if {$msg(type) == "touch.object"} {
		# Update object

		# If empty list of updated fields, means destroy
		# Force unbind, invoke destroy widget event ...

		# array set objectVar would trigger write trace
		disable_trace $objectVar $objectIndex
		array set $objectVar $msg(fields)

		append debug " (update $objectVar to '$msg(fields)')"

		enable_trace $objectVar $objectIndex

		# Remove any of msg(fields) from dirty (and report warning) dirty may not exist ...
	} elseif {$msg(type) == "touch.is"} {
		# Update type (types/instances), assume no trace is set
		array set $objectVar $msg(fields)
		append debug " (update $objectVar to '$msg(fields)')"
	} elseif {$msg(type) == "touch.of"} {
		# Update whole/part, assume no trace is set
		array set $objectVar $msg(fields)
		append debug " (update $objectVar to '$msg(fields)')"
	} elseif {$msg(type) == "touch.in"} {
		# Update place/point, assume no trace is set
		array set $objectVar $msg(fields)
		append debug " (update $objectVar to '$msg(fields)')"
	} elseif {$msg(type) == "touch.has"} {
		# Force unbind, invoke destroy widget event ...
	} else {
		error "unknown publish message type: $msg(type)"
	}
	return $debug
}

}
