
# Msgpack unpacker
# Based on msgpack tcl library by Jos Decoster 
# MessagePack is a compact binary serialization format, supported by Redis/Lua

# Redis/Lua may pack a structure and publish to tcl client 
# (in lua) msgpack = cmsgpack.pack(lua_object)

namespace eval msgpack {

# Unpack msgpack data, return string or list
# String may be a number, list may be in array-get format
# List or array elements may be lists or arrays
proc unpack { data } {
	set result [msgpack::decode data]

	# Should have decoded all the data
	set length [string length $data]
	if {$length} {
		error "msgpack::unpack $length bytes leftover data: \"$data\""
	}
	return $result
}

# Decode msgpack-format data (internal)
# float64 doesn't work on tcl 8.4...
proc decode { dataName } {
	upvar $dataName data

	set length [string length $data]
	if {$length < 1} { error "msgpack::decode no data" }

	binary scan $data c c
	set tc [expr {$c & 0xFF}]
	set data [string range $data 1 end]
	incr length -1

	if {$tc < 0x80} {
		# Positive FixNum
		return [expr {$c & 0x7F}]
	}
	if {($tc & 0xE0) >= 0xE0} {
		# Negative FixNum
		binary scan [binary format c [expr {($c & 0x1F) | 0xE0}]] c c
		return $c
	}
	if {$tc >= 0x80 && $tc <= 0x8F} {
		# FixMap
		set n [expr {$tc & 0xF}]
		set a {}
		for {set i 0} {$i < $n} {incr i} {
			lappend a [msgpack::decode data]
			lappend a [msgpack::decode data]
		}
		return $a
	}
	if {$tc >= 0x90 && $tc <= 0x9F} {
		# FixArray
		set n [expr {$tc & 0xF}]
		set a {}
		for {set i 0} {$i < $n} {incr i} {
			lappend a [msgpack::decode data]
		}
		return $a
	}
	if {$tc >= 0xA0 && $tc <= 0xBF} {
		# FixRaw (alias FixStr)
		set n [expr {$tc & 0x1F}]
		if {$length < $n} { error "FixRaw too short" }
		binary scan $data a$n c
		set data [string range $data $n end]
		return $c
	}
	if {$tc == 0xC0} {
		# nil
		return "nil"
	}
	if {$tc == 0xC2} {
		# false
		return 0
	}
	if {$tc == 0xC3} {
		# true
		return 1
	}
	if {$tc == 0xCA} {
		# float 32
		if {$length < 4} { error "float32 too short" }
		binary scan $data R c
		set data [string range $data 4 end]
		return $c
	}
	if {$tc == 0xCB} {
		if {$length < 8} { error "float64 too short" }
		binary scan $data Q c
		set data [string range $data 8 end]
		return $c
	}
	if {$tc == 0xCC} {
		# uint8
		if {$length < 1} { error "uint8 too short" }
		binary scan $data c c
		set data [string range $data 1 end]
		return [expr {$c & 0xFF}]
	}
	if {$tc == 0xCD} {
		# uint16
		if {$length < 2} { error "uint16 too short" }
		binary scan $data S c
		set data [string range $data 2 end]
		return [expr {$c & 0xFFFF}]
	}
	if {$tc == 0xCE} {
		# uint32
		if {$length < 4} { error "uint32 too short" }
		binary scan $data I c
		set data [string range $data 4 end]
		return [expr {$c & 0xFFFFFFFF}]
	}
	if {$tc == 0xCF} {
		# uint64
		if {$length < 8} { error "uint64 too short" }
		binary scan $data W c
		set data [string range $data 8 end]
		return [expr {$c & 0xFFFFFFFFFFFFFFFF}]
	}
	if {$tc == 0xD0} {
		# int8
		if {$length < 1} { error "int8 too short" }
		binary scan $data c c
		set data [string range $data 1 end]
		return $c
	}
	if {$tc == 0xD1} {
		# int16
		if {$length < 2} { error "int16 too short" }
		binary scan $data S c
		set data [string range $data 2 end]
		return $c
	}
	if {$tc == 0xD2} {
		# int32
		if {$length < 4} { error "int32 too short" }
		binary scan $data I c
		set data [string range $data 4 end]
		return $c
	}
	if {$tc == 0xD3} {
		# int64
		if {$length < 8} { error "int64 too short" }
		binary scan $data W c
		set data [string range $data 8 end]
		return $c
	}
	if {$tc == 0xD9 || $tc == 0xC4} {
		# raw 8 (alias str8 or bin8)
		if {$length < 1} { error "raw8 too short" }
		binary scan $data c c
		set n [expr {$c & 0xFF}]
		set data [string range $data 1 end]
		incr length -1
		if {$length < $n} { error "raw8 too short" }
		binary scan $data a$n c
		set data [string range $data $n end]
		return $c
	}
	if {$tc == 0xDA || $tc == 0xC5} {
		# raw 16 (alias str16 or bin16)
		if {$length < 2} { error "raw16 too short" }
		binary scan $data S n
		set n [expr {$n & 0xFFFF}]
		set data [string range $data 2 end]
		incr length -2
		if {$length < $n} { error "raw16 too short" }
		binary scan $data a$n c
		set data [string range $data $n end]
		return $c
	}
	if {$tc == 0xDB || $tc == 0xC6} {
		# raw 32 (alias str32 or bin32)
		if {$length < 4} { error "raw32 too short" }
		binary scan $data I n
		set n [expr {$n & 0xFFFFFFFF}]
		set data [string range $data 4 end]
		incr length -4
		if {$length < $n} { error "raw32 too short" }
		binary scan $data a$n c
		set data [string range $data $n end]
		return $c
	}
	if {$tc == 0xDC} {
		# array 16
		if {$length < 2} { error "array16 too short" }
		binary scan $data S n
		set n [expr {$n & 0xFFFF}]
		set data [string range $data 2 end]
		set a {}
		for {set i 0} {$i < $n} {incr i} {
			lappend a [msgpack::decode data]
		}
		return $a
	}
	if {$tc == 0xDD} {
		# array 32
		if {$length < 4} { error "array32 too short" }
		binary scan $data I n
		set n [expr {$n & 0xFFFFFFFF}]
		set data [string range $data 4 end]
		set a {}
		for {set i 0} {$i < $n} {incr i} {
			lappend a [msgpack::decode data]
		}
		return $a
	}
	if {$tc == 0xDE} {
		# map 16
		if {$length < 2} { error "map16 too short" }
		binary scan $data S n
		set n [expr {$n & 0xFFFF}]
		set data [string range $data 2 end]
		set a {}
		for {set i 0} {$i < $n} {incr i} {
			lappend a [msgpack::decode data]
			lappend a [msgpack::decode data]
		}
		return $a
	}
	if {$tc == 0xDF} {
		# map 32
		if {$length < 4} { error "map32 too short" }
		binary scan $data I n
		set n [expr {$n & 0xFFFFFFFF}]
		set data [string range $data 4 end]
		set a {}
		for {set i 0} {$i < $n} {incr i} {
			lappend a [msgpack::decode data]
			lappend a [msgpack::decode data]
		}
		return $a
	}

	if {$tc == 0xC7 || $tc == 0xC8 || $tc == 0xC9} {
		# ext 8, 16, 32
		error "ext not supported"
	}
	if {$tc == 0xD4 || $tc == 0xD5 || $tc == 0xD6} {
		# fixext 1, 2, 4
		error "fixext not supported"
	}
	if {$tc == 0xD7 || $tc == 0xD8} {
		# fixext 8, 16
		error "fixext not supported"
	}
	error "msgpack code '$tc' not recognised"
}

}
