
-- Common library functions for Redis scripts

-- Get key for object hash
local getObjectKey = function(domainTag, objectTag)
	return domainTag .. ':' .. objectTag
end

-- Get key for object edit zset
local getObjectEditKey = function(domainTag, objectTag)
	return domainTag .. ':edit.' .. objectTag
end

-- Get key for type/instance zset
local getTypeKey = function(domainTag, objectTag)
	return domainTag .. ':is.' .. objectTag
end

-- Get key for relation zset ('is', 'of', 'in' etc)
local getRelationKey = function(domainTag, relation, objectTag)
	return domainTag .. ':' .. relation .. '.' .. objectTag
end

-- Get key for word zset
local getWordKey = function(domainTag, word)
	return domainTag .. ':word.' .. word
end

-- Get key for prefix zset
local getPrefixKey = function(domainTag, prefix)
	return domainTag .. ':prefix.' .. prefix
end

-- Add alternating name-value pairs to dictionary
local addToSet = function(s, t)
	for i=1, #t, 2 do
		s[t[i]] = t[i+1]
	end
end

-- Convert dictionary to score-name list
local getScoreName = function(s)
	local list = {}
	for name, score in pairs(s) do
		table.insert(list, score)
		table.insert(list, name)
	end
	return list
end

-- Do redis command in chunks (because lua unpack has a size limit)
local redisChunk = function(command, key, members)
	local index = 1
	local remaining = #members
	local chunk = 1000
	while remaining > chunk do
		redis.call(command, key, unpack(members, index, index+(chunk-1)))
		index = index + chunk
		remaining = remaining - chunk
	end
	redis.call(command, key, unpack(members, index))
end

-- Return unique timestamp, increment if necessary
-- Prune recent set on client side, can't easily from lua ...
local getUniqueTimestamp = function(domainTag, timestamp)
		local recentKey = domainTag .. '.recent'
		while true do
			if redis.call('sismember', recentKey, timestamp) == 0 then break end
			timestamp = string.format('%.6f', timestamp + 0.000001)
		end
		redis.call('sadd', recentKey, timestamp)
		return timestamp
end

-- Add existing ancestors (not descendants) to dictionary
-- Ancestors may be types or wholes (ie. containers) 
local addObjectAncestors = function(s, ancestorIsKey)
	local ancestorList = redis.call('zrangebyscore', ancestorIsKey, '-inf', -1, 'withscores')
	for i=1, #ancestorList, 2 do
		local name = ancestorList[i]
		local score = tonumber(ancestorList[i+1])
		s[name] = score
	end
end

-- Get descendants of object (ie. instances or parts)
-- Return list (sorted by score)
local getDescendantList = function(domainTag, relation, objectTag)
	local relationKey = getRelationKey(domainTag, relation, objectTag)
	local descendantList = redis.call('zrangebyscore', relationKey, 1, '+inf', 'withscores')
	if #descendantList == 0 then
		table.insert(descendantList, objectTag)
		table.insert(descendantList, 0)
	end
	return descendantList
end

-- Update object descendants (ie. instances or parts) with new ancestors
-- Update the zsets for descendants, publish changes
local updateDescendants = function(domainTag, relation, objectTag, ancestorList, descendantList)
	-- Merge the updated ancestors into existing descendants
	local payload
	local previous = math.huge
	for i = 1, #descendantList, 2 do
		-- Each descendant of object (and object itself)
		local name = descendantList[i]
		local score = descendantList[i+1]
		if score ~= previous then
			previous = score
			-- Decrement scores for ancestors
			for i = 1, #ancestorList, 2 do
				ancestorList[i] = ancestorList[i]-1
			end

			-- Update payload to be published
			local object = {}
			for i = 1, #ancestorList, 2 do
				local score = ancestorList[i]
				local name = ancestorList[i+1]
				object[name] = score
			end
			-- Everybody process published change (including originator)
			-- Include instance, client should only ignore 'touch.object' payloads with same instance ...
			-- type 'touch.{relationTag}' change redis_interface.tcl ...
			local operationType = 'touch.'.. relation
			local operation = {type=operationType, instance='', fields=object}
			payload = cmsgpack.pack(operation)
		end

		-- Update descendant with ancestors
		local relationKey = getRelationKey(domainTag, relation, name)
		redisChunk('zadd', relationKey, ancestorList)

		-- publish the change to object descendants
		redis.call('publish', relationKey, payload)
	end
end

-- Update object ancestors (ie. types or wholes) with new descendants
-- Update the zsets for ancestors, publish changes
local updateAncestors = function(domainTag, relation, objectTag, ancestors, descendantList)
	-- Get ancestors sorted by score (closeness of relation)
	local sortedAncestors = {}
	for name, score in pairs(ancestors) do
		table.insert(sortedAncestors, name)
	end
	table.sort(sortedAncestors, function(a, b) return ancestors[a] > ancestors[b] end)

	-- create descendantArray (score-member)
	local descendantArray = {}
	for i = 1, #descendantList, 2 do
		local name = descendantList[i]
		local score = descendantList[i+1]
		table.insert(descendantArray, score)
		table.insert(descendantArray, name)
	end

	-- Merge object descendants (including object itself) into ancestors
	local payload
	local previous = math.huge
	for i, name in ipairs(sortedAncestors) do
		-- Each ancestor (not including object itself)
		local score = ancestors[name]
		if score ~= previous then
			previous = score
			-- Increment scores for descendants
			for i = 1, #descendantArray, 2 do
				descendantArray[i] = descendantArray[i]+1
			end

			-- Update payload to be published
			local object = {}
			for i = 1, #descendantArray, 2 do
				local score = descendantArray[i]
				local name = descendantArray[i+1]
				object[name] = score
			end
			-- Everybody process published change (including originator)
			-- Include instance, client should only ignore 'touch.object' payloads ...
			local operationType = 'touch.'.. relation
			local operation = {type=operationType, instance='', fields=object}
			payload = cmsgpack.pack(operation)
		end

		-- Update ancestors with descendants
		local relationKey = getRelationKey(domainTag, relation, name)
		redisChunk('zadd', relationKey, descendantArray)

		-- publish the change to object ancestors
		redis.call('publish', relationKey, payload)
	end
end

-- Split value of name fields into words
-- Don't publish changes to keywords
-- For all name fields, not just different ones ...
-- Split a word "abc123" or "123abc" into two words, add whole and splits ...
local updateKeywords = function(domainTag, objectTag, names, different)
	local words = {}
	local ignoreWords = {
		['ago'] = 1, ['all'] = 1, ['am'] = 1, ['an'] = 1, ['and'] = 1, ['any'] = 1, ['are'] = 1,
		['as'] = 1, ['at'] = 1, ['be'] = 1, ['by'] = 1, ['did'] = 1, ['do'] = 1, ['for'] = 1,
		['get'] = 1, ['go'] = 1, ['got'] = 1, ['had'] = 1, ['has'] = 1, ['he'] = 1, ['if'] = 1,
		['in'] = 1, ['is'] = 1, ['it'] = 1, ['its'] = 1, ['me'] = 1, ['my'] = 1, ['no'] = 1,
		['of'] = 1, ['oh'] = 1, ['on'] = 1, ['or'] = 1, ['our'] = 1, ['so'] = 1, ['the'] = 1,
		['to'] = 1, ['too'] = 1, ['with']  = 1
	}
	for name, priority in pairs(names) do
		local value = different[name]
		-- Split value into words (alphanumeric)
		for word in value:gmatch('%w+') do
			local length = string.len(word)
			if length > 1 and length < 15 then
				word = string.lower(word)
				-- Ignore overly common words
				if ignoreWords[word] == nil then
					-- Add word to dictionary, use max priority
					if words[word] then
						if priority > words[word] then
							words[word] = priority
						end
					else
						words[word] = priority
					end
				end
			end
		end
	end

	for word, priority in pairs(words) do
		-- Add object to each words' zset, use max priority
		local wordKey = getWordKey(domainTag, word)
		local oldPriority = tonumber(redis.call('zscore', wordKey, objectTag))
		if not oldPriority or priority > oldPriority then
			redis.call('zadd', wordKey, priority, objectTag)
		end

		-- Reduce word to prefixes
		local prefix = word
		local length = string.len(word)
		while length > 1 do
			-- Add word to each prefix zset
			local prefixKey = getPrefixKey(domainTag, prefix)
			redis.call('zincrby', prefixKey, 0, word)

			length = length -1
			prefix = string.sub(prefix, 1, length)
		end
	end
end

-- Identify each field as:
-- is.<type>			type of thing, party or event
-- name					given to subject, value is the name (update keywords)
-- name.<role>			name given to subject per role
-- of.<whole>			subject is part of a whole
-- in.<place>			subject is a point within a place
-- has.<property>		instance of subject-type must have property (per qualifier)
-- <property>			subject has property, value is string, number or tag
-- <property>.<role>	property attributed to subject per entity or role
local characteriseFields = function(domainTag, subjectTag, fields)

	local subjectTypes = {}
	local subjectWholes = {}
	local subjectPlaces = {}
	local names = {}

	redis.log(redis.LOG_NOTICE, 'characteriseFields ' .. subjectTag)

	for name, value in pairs(fields) do
		-- Parse field name, extract relation label and tag
		local dot = string.find(name, '.', 1, true)
		if dot then
			local label = string.sub(name, 1, dot-1)
			local tag = string.sub(name, dot+1)
			if label == 'is' then
				-- is.<type>  subject is instance of type (update type/instance zset recursively)
				local typeKey = getTypeKey(domainTag, tag)
				addObjectAncestors(subjectTypes, typeKey)
				subjectTypes[tag] = 0
			elseif label == 'name' then
				-- name.<party>  name given to subject by party
				-- Get priority/weight for name.<tag>
				local objectKey = getObjectKey(domainTag, name)
				local priority = redis.call('hget', objectKey, 'priority') or 0
				names[name] = priority
			elseif label == 'of' then
				-- of.<whole>  subject is part of whole
				-- Get whatever this field is part of
				local wholeKey = getRelationKey(domainTag, 'of', tag)
				addObjectAncestors(subjectWholes, wholeKey)
				subjectWholes[tag] = 0
			elseif label == 'in' then
				-- in.<place>  subject is a point within a place
				-- Get whatever this field is within
				local placeKey = getRelationKey(domainTag, 'in', tag)
				addObjectAncestors(subjectPlaces, placeKey)
				subjectPlaces[tag] = 0
			elseif label == 'has' then
				-- has.<property>  instance of subject-type must have property (per qualifier)
				--
			else
				-- <property>.<party>  property attributed to subject by party
				--
			end
		else
			-- no dot
			if name == 'name' then
				-- name given to subject, value is the name (update prefix/word zsets)
				names[name] = 10
			elseif name == 'description' then
				-- description given to subject, value is the text (update prefix/word zsets)
				names[name] = 5
			else
				-- subject has property, value is string, number or tag
			end
		end
	end

	local typeList = getScoreName(subjectTypes)
	if #typeList > 0 then
		-- Apply changes to subject instances and types, publish changes
		local relation = 'is'
		local instanceList = getDescendantList(domainTag, relation, subjectTag)
		updateDescendants(domainTag, relation, subjectTag, typeList, instanceList)
		updateAncestors(domainTag, relation, subjectTag, subjectTypes, instanceList)
	end

	local wholeList = getScoreName(subjectWholes)
	if #wholeList > 0 then
		-- Apply changes to subject wholes and parts, publish changes
		local relation = 'of'
		local partList = getDescendantList(domainTag, relation, subjectTag)
		updateDescendants(domainTag, relation, subjectTag, wholeList, partList)
		updateAncestors(domainTag, relation, subjectTag, subjectWholes, partList)
	end

	local placeList = getScoreName(subjectPlaces)
	if #placeList > 0 then
		-- Apply changes to subject places and points, publish changes
		local relation = 'in'
		local pointList = getDescendantList(domainTag, relation, subjectTag)
		updateDescendants(domainTag, relation, subjectTag, placeList, pointList)
		updateAncestors(domainTag, relation, subjectTag, subjectPlaces, pointList)
	end

	-- Update prefix and word zsets for words in names
	updateKeywords(domainTag, subjectTag, names, fields)
end

