
-- Redis script to edit an object
-- Update object including types and instances, publish changes
-- Create an edit (if object is tracked)
-- Return tag of edit (if an edit was created)
-- Arguments:
--   domainTag     tag of database domain
--   objectTag     tag of object we're updating
--   sessionTag    tag of user session
--   instanceID    unique client-instance identifier
--   timestamp     time at client (unix time to usec)
--   {name value}  list of new/updated fields

local domainTag = KEYS[1]
local objectTag = KEYS[2]
local sessionTag = KEYS[3]
local instanceID = KEYS[4]
local timestamp = KEYS[5]

-- Get existing fields (as a dictionary)
local objectKey = getObjectKey(domainTag, objectTag)
local existingList = redis.call('hgetall', objectKey)
local existing = {}
addToSet(existing, existingList)

-- Get the objects' existing types (not instances)
-- Or get 'tracked' from characteriseFields ...
local objectIsKey = getTypeKey(domainTag, objectTag)
local objectIs = {}
addObjectAncestors(objectIs, objectIsKey)

-- Determine fields that have actually changed
local differentList = {}
for i = 1, #ARGV, 2 do
	local name = ARGV[i]
	local newValue = ARGV[i+1]
	local oldValue = existing[name] or ''
	if newValue ~= oldValue then
		table.insert(differentList, name)
		table.insert(differentList, newValue)
	end
end

local editTag = ''

-- Apply change to object
-- Empty argument list = delete object
if #ARGV == 0 or #differentList > 0 then
	local different = {}
	addToSet(different, differentList)

	-- Identify types, properties, parts, names; update zset indexes
	characteriseFields(domainTag, objectTag, different)

	-- Update or delete object fields
	for name, value in pairs(different) do
		if value ~= '' then
			redis.call('hset', objectKey, name, value)
		else
			redis.call('hdel', objectKey, name)
		end
	end

	-- Empty list of new/updated fields, delete the object
	if #ARGV == 0 then
		redis.call('del', objectKey)
	end

	-- Publish change to object (originator will ignore)
	local operation = {type='touch.object', instance=instanceID, fields=different}
	local payload = cmsgpack.pack(operation)
	redis.call('publish', objectKey, payload)

	redis.log(redis.LOG_NOTICE, "edit: publish " .. objectKey)

	-- Create edit (that may be undone)
	if objectIs['tracked'] then
		-- Allocate tag for edit
		editTag = getUniqueTimestamp(domainTag, timestamp)
		local editKey = getObjectKey(domainTag, editTag)

		-- Create edit (only if object already existed)
		if #existingList > 0 then
			local editList = {}
			for name, newValue in pairs(different) do
				local oldValue = existing[name] or ''
				table.insert(editList, name)
				table.insert(editList, oldValue)
			end
			redisChunk('hmset', editKey, editList)
		end

		-- Add edit to set of object edits
		local score = sessionTag
		if score == '' then score = 0 end
		local objectEditKey = getObjectEditKey(domainTag, objectTag)
		redis.call('zadd', objectEditKey, score, editTag)
	end
end

-- return tag of edit or ''
return {ok=editTag}

