
# Database queries, using proxy
# Replacements for the sql namespace procedures

namespace eval query {

# Perform database query/statement
# Arguments and options:
#   row            Name of array where row-result returned
#   table          Database table name (only use if query is on a single table)
#   statement      SQL statement to execute
#   action         Invoke action on each row returned
proc forevery { args } {
	# Get arguments and options
	set defaults {table ""}
	command::arguments $args opt {row statement action} $defaults
	global setup
	global errorInfo

	# Where row result returned
	upvar $opt(row) resultrow
	set table $opt(table)
	set statement $opt(statement)
	set action $opt(action)

	# Stack trace for fingerprints
	set context [get_calling_context "::query::forevery"]

	# Flush cache, don't clear; assume this is a read-query (support list of tables ...)
	query::flush_cache $table

	# Connected via fbclient
	if {[info exists setup(remote_host)] && $setup(remote_host) != ""} {
		if {$::fbremote::disconnected} { offer_reconnect_remote }
		if {[catch { set result [fbremote::sql_array $statement $context] } msg]} {
			if {$::fbremote::disconnected} { offer_quit_remote $errorInfo }
			error $errorInfo
		}

		# Row only includes the changed fields
		array set rowArray {}
		foreach row $result {
			# Copy complete array to caller
			array set rowArray $row
			array set resultrow [array get rowArray]

			# 0 = ok, 1 = error, 2 = return, 3 = break, 4 = continue
			set code [catch { uplevel 1 $action } msg]
			switch -- $code {
				0 { }
				1 { return -code error \
							-errorinfo $::errorInfo \
							-errorcode $::errorCode $msg }
				2 { return -code return $msg }
				3 { return {} }
				4 { }
				default { return -code $code $msg }
			}
		}
		return ""
	}

	# Connected via fbproxy
	set connID $::sql::connID
	if {$connID == ""} { error "forevery: not connected" }
	set id [proxy::sendArrayQuery $connID $statement $context]
	array set rowArray {}
	set final 1

	# Get responses
	while true {
		# Get block of response rows
		set response [proxy::receive $connID $id]
		set type [lindex $response 0]
		switch -- $type {
			"ok" {
				proxy::requestComplete $connID $id $final
				error "unexpected ok from proxy"
			}
			"error" {
				set reasonCode [lindex $response 1]
				set reasonString [lindex $response 2]
				proxy::requestComplete $connID $id $final
				error "$reasonCode $reasonString"
			}
			"array" {}
			default {
				proxy::requestComplete $connID $id $final
				error "expected array or error from proxy"
			}
		}

		set final [lindex $response 1]
		set block [lindex $response 2]

		# Row only includes the changed fields
		foreach row $block {
			# Copy complete array to caller
			array set rowArray $row
			array set resultrow [array get rowArray]

			# 0 = ok, 1 = error, 2 = return, 3 = break, 4 = continue
			set code [catch { uplevel 1 $action } msg]
			switch -- $code {
				0 { }
				1 { proxy::requestComplete $connID $id $final
					return -code error \
							-errorinfo $::errorInfo \
							-errorcode $::errorCode $msg }
				2 { proxy::requestComplete $connID $id $final
					return -code return $msg }
				3 { proxy::requestComplete $connID $id $final
					return {} }
				4 { }
				default { proxy::requestComplete $connID $id $final
						  return -code $code $msg }
			}
		}
		if {$final} { break }
	}

	proxy::requestComplete $connID $id $final
	return ""
}

# Return number of rows affected by the last query (insert, update or delete)
# Result must be the return value of the sql command
proc affected_rows { result } {
	global setup
	
	if {[info exists setup(remote_host)] && $setup(remote_host) != ""} {
		if {[namespace exists ::fbremote] && $::fbremote::version2 == 0} {
			return [sql numrows]
		}
	}
	if {$::sql::fbproxy} {
		if {[llength $result] < 2} {
			error "affected_rows: expecting list parameter"
		}
		return [lindex $result 0]
	} else {
		return [sql numrows]
	}
}

# Return the insert ID from the last query (insert)
# Result must be the return value of the sql command
proc insert_id { result } {
	global setup

	if {[info exists setup(remote_host)] && $setup(remote_host) != ""} {
		if {[namespace exists ::fbremote] && $::fbremote::version2 == 0} {
			return [lindex [lindex [sql "select LAST_INSERT_ID()"] 0] 0]
		}
	}
	if {$::sql::fbproxy} {
		set insertID [lindex $result 1]
		if {$insertID == ""} {
			error "insert_id must come from a previous 'sql insert'"
		}
		return $insertID
	} else {
		return [lindex [lindex [sql "select LAST_INSERT_ID()"] 0] 0]
	}
}

}

# -------------------------------------------------------------------------

# Create placeholder sql variables 
if {! [info exists ::sql::fbproxy]} {
	namespace eval sql {
		variable fbproxy 1
	}
}

