
# Connect to proxy server and access it's services
# Part of the fbproxy project - replacement for fbsql.so and fbsql.dll
# Provides database query services required by softsql.tcl and sql.tcl

# Improvements
#   Client is entirely TCL hence more portable and easier to understand
#   Results may be returned in array or list form, for all queries
#   Multiple simultaneous requests are allowed
#   Response may be received in any order, so parallel-processing is possible
#   Detects when connection is closed.  Reliable auto-reconnect is possible
#   A query in-progress may be cancelled

# Limitations
# Sessions are pooled by fbproxy, so you must use "stateless sql".  Dont use:
#   "sql numrows" - instead use return value from the query
#   "select last_insert_id" - instead use return value from the query
#   "lock table" - instead use select/update in a loop
#   "set @variable" (user defined variables) - instead use client-side logic
#   "show session status" or "flush status" - won't be very useful
#   "flush tables with read lock" - don't use
#   "begin transaction" - use application-level consistency checking
#   "create temporary table" - use client-side array or list
#   "use database" - select database when connecting

# ------------------------------------------------------------------------------------
# TODO

# Define a bgerror procedure
# Interaction with progress bar, and the update command
# Request email body, or printable file, or image put/get...
# Auth - send client computer name and user, for logging...

# ------------------------------------------------------------------------------------
# Protocol spec

# Header - list followed by newline
#   length  - integer, length of body in bytes
#   id - integer, request id (per connection) (id 0 = ping)

# Request body
#   A list, first element is type; auth, selectdb, query, cancel
#   auth - sql_user, sql_password, sql_database, client_computer, client_user, date
#   selectdb - sql_database
#   query - coding 0/1 (list or array), statement, context (optional)
#   cancel - no other params
# Response body
#   A list, first element is type; ok, error, list or array
#   ok - affected_rows, insert_id, expiry_date (for auth) 
#   error - reason (number), reason (string)
#   list - final (0/1), result in list-of-list form
#   array - final (0/1), result in list-of-array form

namespace eval proxy {

# ---------------------------------------------------------------------------------
# tcp connection management
# connect to server, disconnect, re-connect
# supports multiple connections; softsql uses a single connection to one server

# State per connection - indexed by sockID - created on connect - deleted on disconnect
# connected - 0/1
# required - bytes of data to come, from header
# requestID - from header
# buffer - partially received data
# lastID - last request ID
# lastError - string
# ping - ping after ID - split into pingID and pingTime ...

# State per request - indexed by sockID,requestID - deleted on final response
# queue - list of responses received

# ---------------------------------------------------------------------------------

# Make a new connection to a proxy server
# Return socket ID
proc connect { hostname port } {
	variable connected
	variable required
	variable requestID
	variable buffer
	variable lastID
	variable lastError
	variable ping
	variable queue

	if {$port == ""} { set port 6468 }

	# Improve the socket error message (usually a DNS problem)
	if {[catch {set sock [socket -async $hostname $port]} msg]} {
		set reason ""
		regexp {^[^:]*: *(.*)$} $msg all reason
		if {$reason == "invalid argument"} { set reason "host is unreachable" }
		error "cannot connect to $hostname port $port:\n$reason"
	}

	# New connection state
	set connected($sock) 0
	set required($sock) 0
	set requestID($sock) 0
	set buffer($sock) ""
	set lastID($sock) 0
	set lastError($sock) ""
	set ping($sock) ""
	array set queue {}

	# Attempt connect
	# use default translation and encoding, for unicode/utf-8 support ...
	# still don't want cr or lf translation ...
	# socket_writable is not called on Windows7 if McAfee is enabled (pudney)  ...
	fconfigure $sock -buffering full -translation binary -buffersize 65536 -blocking 0
	fileevent $sock writable [list ::proxy::socket_writable $sock]
	vwait ::proxy::connected($sock)
	if {! $connected($sock)} {
		error "cannot connect to $hostname port $port:\n$lastError($sock)"
	}

	# Connected
	fileevent $sock writable {}
	fileevent $sock readable [list ::proxy::socket_readable $sock]
	set ping($sock) [after 5000 [list ::proxy::send_ping $sock]]
	return $sock
}

# Disconnect connection
proc disconnect { sock } {
	variable connected
	variable required
	variable requestID
	variable buffer
	variable lastID
	variable lastError
	variable ping
	variable queue

	if {$sock == ""} { return }
	if {! [info exists connected($sock)]} { return }
	if {! $connected($sock)} { return }

	catch { close $sock }
	if {$ping($sock) != ""} {
		after cancel $ping($sock)
		set ping($sock) ""
	}
	set connected($sock) 0

	# Login has disconnect-handling behaviour
	if {[info commands ::login::disconnected] != ""} {
		set reconnect [expr {[llength [array names queue]] == 0}]
		login::disconnected $sock $lastError($sock) $reconnect
	}

	# Wakeup receivers
	foreach index [array names queue "$sock,*"] {
		if {$queue($index) == ""} {
			set queue($index) {}
		}
	}
}

# Connection socket is writable, check if connected OK
proc socket_writable { sock } {
	variable connected
	variable lastError

	set msg [fconfigure $sock -error]
	if {$msg != ""} {
		# Connect failed, e.g. "connection refused", "host is unreachable"
		set lastError($sock) $msg
		catch { close $sock }
		set connected($sock) 0
		return
	}
	set connected($sock) 1
}

# Read partial or complete response data from socket
# Also called when socket is closed by server
proc socket_readable { sock } {
	variable required
	variable requestID
	variable buffer
	variable queue
	variable lastError
	variable ping

	# Waiting for header
	if {$required($sock) == 0} {
		set result [gets $sock header]
		if {$result < 0} {
			if {[eof $sock]} {
				set lastError($sock) "connection $sock has been closed"
				disconnect $sock
			}
			return
		}
		# Received header
		set required($sock) [lindex $header 0]
		set requestID($sock) [lindex $header 1]
		set buffer($sock) ""
		if {![string is integer -strict $required($sock)] || ![string is integer -strict $requestID($sock)]} {
			set lastError($sock) "invalid header data from server"
			disconnect $sock
			return
		}
	}

	# EOF after header
	if {[eof $sock]} {
		set lastError($sock) "truncated data from server"
		disconnect $sock
		return
	}

	# More data to receive
	if {$required($sock) > 0} {
		set data [read $sock $required($sock)]
		set received [string length $data]

		append buffer($sock) $data
		set required($sock) [expr {$required($sock) - $received}]
		# Need more still ?
		if {$required($sock) > 0} {
			return
		}
	}
	# Received complete message

	# Reset send-ping time
	if {$ping($sock) != ""} {
		after cancel $ping($sock)
	}
	set ping($sock) [after 30000 [list ::proxy::send_ping $sock]]

	# Ping response
	if {$requestID($sock) == 0} { return }

	set index "$sock,$requestID($sock)"

	# Append new response to queue
	if {[info exists queue($index)]} {
		lappend queue($index) $buffer($sock)
	} else {
		# Nothing waiting, discard response
	}
	set required($sock) 0
	set requestID($sock) 0
	set buffer($sock) ""
}

# Write data to socket, with "length id newline" prefix
proc transmit { sock id data } {
	variable connected
	variable lastError

	if {! $connected($sock)} {
		if {$lastError($sock) != ""} {
			error "cannot send query: $lastError($sock)"
		}
		error "proxy::transmit: not connected"
	}
	set length [string length $data]
	puts $sock [list $length $id]
	puts -nonewline $sock $data
	flush $sock
}

# Send ping message to server
proc send_ping { sock } {
	variable ping

	transmit $sock 0 ""
}

# Allocate a request ID
proc allocateID { sock } {
	variable lastID
	variable queue

	set id [incr lastID($sock)]
	set index "$sock,$id"

	set queue($index) {}
	return $id
}

# Cancel request, destroy request/response state
proc requestComplete { sock id final } {
	variable queue

	set index "$sock,$id"
	if {! [info exists queue($index)]} { return }

	# Most recent response received
	set response [lindex $queue($index) end]
	set type [lindex $response 0]
	if {$type == "list" || $type == "array"} {
		set final [lindex $response 1]
	}
	# Send cancel if more responses are expected
	if {! $final} { transmit $sock $id "cancel" }
	unset queue($index)
}

# Receive response for this request ID
# Return received data
# Allow timeout, on reader raise error & delete var, on response fail silently if var doesn't exist...
# vwait creates a nested event loop, issues for parallel ...
proc receive { sock id } {
	variable connected
	variable queue

	# Wait for data
	set index "$sock,$id"
	while {[llength $queue($index)] == 0} {
		if {! $connected($sock)} {
			requestComplete $sock $id 1
			if {$lastError($sock) != ""} {
				error "cannot receive response: $lastError($sock)"
			}
			error "proxy::receive: not connected"
		}
		vwait ::proxy::queue($index)
	}

	# Consume oldest response
	set result [lindex $queue($index) 0]
	# Faster version of [lrange $queue($index) 1 end]
	set queue($index) [lreplace $queue($index) [set queue($index) 0] 0]
	return $result
}

# Connect to proxy, send mysql authenticate request, receive response
# Raise error if connection fails or authentication fails
proc connectdb { proxy_host sql_user sql_password sql_database proxy_port client_computer client_user client_date} {
	set sock [connect $proxy_host $proxy_port]
	set id [allocateID $sock]
	set request [list "auth" $sql_user $sql_password $sql_database $client_computer $client_user $client_date]
	transmit $sock $id $request

	set response [receive $sock $id]
	requestComplete $sock $id 1
	set type [lindex $response 0]
	if {$type == "error"} {
		set reasonCode [lindex $response 1]
		set reasonString [lindex $response 2]
		error "$reasonCode $reasonString"
	}
	set date ""
	if {$type == "ok"} {
		set date [lindex $response 3]
	}
	return [list $sock $date]
}

# Select sql database
proc selectdb { sock sql_database } {
	set id [allocateID $sock]
	set request [list "selectdb" $sql_database]
	transmit $sock $id $request

	set response [receive $sock $id]
	requestComplete $sock $id 1
	set type [lindex $response 0]
	if {$type == "error"} {
		set reasonCode [lindex $response 1]
		set reasonString [lindex $response 2]
		error "proxy::selectdb failed: $reasonCode $reasonString"
	}
}

# Cancel previous query - no response
proc cancel { sock id } {
	set request [list "cancel"]
	transmit $sock $id $request
}

# Send database query/statement to server, expect response in list format
proc sendListQuery { sock statement context } {
	set id [allocateID $sock]
	set request [list "query" 0 $statement $context]
	transmit $sock $id $request
	return $id
}

# Send database query/statement to server, expect response in array format
# Each row only has the fields that are different from the previous row
proc sendArrayQuery { sock statement context } {
	set id [allocateID $sock]
	set request [list "query" 1 $statement $context]
	transmit $sock $id $request
	return $id
}

}
