
# Database queries in parallel
# Part of query namespace
# Use or clear cache ...
# Use tcl8.6 coroutines ...

namespace eval query {

# Perform database query, execute instances of the command in parallel
# command execution is partially-ordered
proc parallel { statement command } {
	variable completed

	set connID $::sql::connID
	set context [get_calling_context "::query::parallel"]
	set id [proxy::sendArrayQuery $connID $statement $context]
	array set rowArray {}
	set final 1

	set script [namespace code $command]
	set index "$connID,$id"
	set completed($index) 0
	array set rowArray {}
	set rowNo 0

	# Get responses
	while true {
		# Get block of response rows
		set response [proxy::receive $connID $id]
		set type [lindex $response 0]
		switch -- $type {
			"error" {
				set reasonCode [lindex $response 1]
				set reasonString [lindex $response 2]
				proxy::requestComplete $connID $id $final
				error "$reasonCode $reasonString"
			}
			"array" {}
			default {
				proxy::requestComplete $connID $id $final
				error "expected array or error from proxy"
			}
		}
		set final [lindex $response 1]
		set block [lindex $response 2]

		# Row from server only includes the changed fields
		foreach row $block {
			array set rowArray $row
			incr rowNo

			# Wait until OK to process this row
			while {[expr {$rowNo - $completed($index)}] > 7} {
				vwait ::query::completed($index)
			}

			# Process row in the background
			after 0 [list query::processParallelRow $index $rowNo $script [array get rowArray]]
		}
		if {$final} { break }
	}

	# Wait until all rows completed
	while {$completed($index) < $rowNo} {
		vwait ::query::completed($index)
	}
	unset completed($index)

	proxy::requestComplete $connID $id $final
	return ""
}

# Called by a parallel-command
# Switch to synchronous operation for the remainder of the command
proc sync { } {
	variable completed

	# Get rowNo and index from procedure stack
	set level 1
	while true {
		if {[catch {upvar $level parallelRowNo rowNo}]} {
			error "must be called from a parallel-row command"
		}
		if {[info exists rowNo]} { break }
		incr level
	}
	upvar $level parallelIndex index

	# Wait for previous command instance to complete
	while {[expr {$rowNo - $completed($index)}] > 1} {
		vwait ::query::completed($index)
	}
}

# Execute parallel command
proc processParallelRow { index rowNo script rowData } {
	variable completed

	# Set local vars needed by sync command
	set parallelIndex $index
	set parallelRowNo $rowNo

	# Execute script (command in original namespace)
	# 0 = ok, 1 = error, 2 = return, 3 = break, 4 = continue
	set code [catch { eval [list $script $rowData] } msg]
	switch -- $code {
		0 { }
		1 { logToFile "$::errorInfo $msg" }
	}

	# Wait for previous command to complete
	while {[expr {$rowNo - $completed($index)}] > 1} {
		vwait ::query::completed($index)
	}
	set completed($index) $rowNo
}

}

