
# support procedures for master keywords

namespace eval keyword {

#-----------------------------------# 'PUBLIC' PROCEDURES #-------------------------------#

# update keywords for a master account
# account details must be provided via -master_name or -module & -number (& -company)
# uses procedures defined in sql.tcl
#
# possible		possible values
# args:			(* = default):
#
# -master_name	name of master array in calling procedure
# -company		company id of master account, default is login::company_id
# -module		module of master account
# -number		number of master account
# -part			external number of master account (can be used instead of number)
# -action		update*,insert,delete

proc update_keywords { args } {
	variable company_id
	variable module
	variable number
	variable master
	variable master_keywords
	variable master_keyword_fields
	# last_sql_statement - used by calling namespaces for error reporting, e.g. see update.tcl
	variable last_sql_statement
	
	# Maintain keywords for master accounts, used for improving search efficiency
	if { ![setup master_keywords 1] } { return }

	# process options
	array set options {master_name "" company "" module "" number "" part "" action "update"}
	foreach { option value } $args {

		set option [string range $option 1 end]
		switch -- $option {
			default { set options($option) $value }
		}
	}

	# load master array
	if { $options(master_name) != "" } {
		upvar $options(master_name) master_array
		array set master [array get master_array]
		set company_id $master(COMPANY_ID)
		set module $master(MODULE)
		set number $master(NUMBER)

	} else {
		set company_id $options(company)
		if { $company_id == "" } { set company_id $login::company_id }

		set module $options(module)

		if { $options(number) != "" } {
			set number $options(number)
			sql::find master "COMPANY_ID = '$company_id' and MODULE = '$module' and NUMBER = $number"
		} else {
			set part_num [sql::escape_string $options(part)]
			sql::find master "COMPANY_ID = '$company_id' and MODULE = '$module' and PART = $part_num"
			set number $master(NUMBER)
		}
	}

	if { $options(action) != "insert" } { delete_keywords }
	if { $options(action) == "delete" } { return }

	# build lists of keywords, in an array indexed by priority
	array unset master_keywords
	array set master_keywords {}

	foreach field_list $master_keyword_fields {
		set field [lindex $field_list 0]
		set arguments [lrange $field_list 1 end]

		
		set priority 9
		set modules {}
		set apply_ignore_words 0
		set field_table "MASTER"
		foreach { option value } $arguments {
			set option [string trimleft $option -]
			switch -- $option {
				"priority" { set priority $value }
				"modules" { set modules $value }
				"ignore_words" { set apply_ignore_words $value }
				"table" { set field_table $value }
			}
		}

		if { $modules != {} && [lsearch $modules $module] < 0 } { continue }

		# not the MASTER table?
		if { $field_table != "MASTER" } {
			set field_value [load_field_value $field_table $field]
			set field_keywords [string_keywords $field_value $apply_ignore_words]
		} else {
			set field_keywords [string_keywords $master($field) $apply_ignore_words]
		}
		
		store_keywords $field_keywords $priority
	}

	# get module-specific keywords
	switch -- $master(MODULE) {
		"AP" { supplier_keywords }
		"AR" { customer_keywords }
		"IN" { product_keywords }
		"GL" { gl_keywords }
		"FA" { asset_keywords }
	}

	#store keywords: build up values to insert, do insert in one transaction
	set insert_rows {}
	array set used_keywords {}
	foreach priority [lsort -integer [array names master_keywords]] {
		foreach keyword $master_keywords($priority) {

			# discard duplicates
			if { [info exists used_keywords($keyword)] } { continue }
			set used_keywords($keyword) 1

			lappend insert_rows "'$master(COMPANY_ID)','$master(MODULE)',$master(NUMBER),'$keyword',$priority"

			# if we are inserting more than 100 rows, break it up into chunks
			if { [llength $insert_rows]>=100 } {
				set last_sql_statement "insert into MASTER_KEYWORD (COMPANY_ID,MODULE,NUMBER,KEYWORD,PRIORITY) values([join $insert_rows "),("])"
				sql $last_sql_statement

				set insert_rows {}
			}
		}
	}

	# no keywords?
	if { $insert_rows == {} } { return }

	set last_sql_statement "insert into MASTER_KEYWORD (COMPANY_ID,MODULE,NUMBER,KEYWORD,PRIORITY) values([join $insert_rows "),("])"
	sql $last_sql_statement
}

# Rebuild ALL keywords for a company
# calling procedure must provide company id if login array does not exist 
proc rebuild_keywords { {company_id ""} } {
	variable last_sql_statement

	if { $company_id == "" } {
		set company_id $login::company_id
	}

	# some setup will need to be done if we are running outside of fastbase environment
	if { [array names sql::tables "all"] == {} } {
		sql::setup_tables
	}

	# delete ALL keywords
	set last_sql_statement "delete from MASTER_KEYWORD where COMPANY_ID = '$company_id'"
	sql $last_sql_statement

	# process all master accounts
	sql::forevery master "COMPANY_ID = '$company_id'" {
		update_keywords -master_name master -action insert
	}
}

# remove ignore words, separators, non-alphanums and trailing whitespace. Return cleaned text
proc clean_text { text } {
	variable separators

	set words {}
	foreach word [split $text $separators] {
		# remove any non-alphanumerics:
		regsub -all {[^A-Z|a-z|0-9]} $word "" word
		if { $word == "" } { continue }
		if { [is_ignore_word $word] } { continue }

		lappend words $word
	}

	return [join $words]
}

# return a list of keywords derived from text
proc string_keywords { text { apply_ignore_words 0 }} {
	variable separators
	variable minimum_word_length
	variable maximum_word_length

	set keywords {}

	foreach word [split $text $separators] {
		regsub -all {[^A-Z|a-z|0-9]} $word "" word
		set word [string tolower $word]

		if { $word == "" } { continue }
		if { $apply_ignore_words && [is_ignore_word $word] } { continue }

		set words [list $word]

		# check if word is a combination of letters and numbers
		if { [regexp {^([a-z]+)([0-9].*)} $word match prefix secondKeyword] } {
			# leading letters with a substring of numbers
			lappend words $prefix $secondKeyword
		} elseif { [regexp {^([0-9]+)([a-z].*)} $word match prefix secondKeyword] } {
			# leading numbers with a substring of letters
			lappend words $prefix $secondKeyword
		}

		foreach keyword $words {
			# check length
			set length [string length $keyword]
			if { $length < $minimum_word_length } { continue }

			if { $length < $maximum_word_length } {
				set keyword [string range $keyword 0 [expr {$maximum_word_length - 1}]]
			}

			lappend keywords $keyword
		}
	}

	return $keywords
}

# return a list of contracted keywords derived from a single word
proc contractions_of { word } {
	variable minimum_contraction
	variable maximum_contraction

	set keywords {}
	set keyword [string tolower $word]
	set length [string length $keyword]

	# maximum contraction?
	if { $length > $maximum_contraction} {
		set keyword [string range $keyword 0 $maximum_contraction]
		set length [expr {$maximum_contraction + 1}]

		if { ![is_ignore_word $keyword] } {
			lappend keywords $keyword
		}
	}

	# add each contraction to the list of keywords
	while { [incr length -1] >= $minimum_contraction } {
		set keyword [string range $keyword 0 end-1]
		if { [is_ignore_word $keyword] } { continue }

		lappend keywords $keyword
	}

	return $keywords
}

# given a database field for the MASTER table, e.g. "NAME, PHONE_NO, SELL_PRICE1"
# return 1 if keywords are derived from the field, i.e. keywords have to be regenerated if the field is modified.
proc is_keyword_field { field {module ""} } {
	variable master_keyword_fields

	foreach field_list $master_keyword_fields {
		set field_name [lindex $field_list 0]
		set field_args [lrange $field_list 1 end]
		set field_options(modules) {}
		set field_options(table) "MASTER"
		command::arguments $field_args field_options {}

		# if the keyword field is not for the MASTER table, ignore
		if { $field_options(table) != "MASTER" } { continue }
		
		# is this the field we're looking for?
		if { $field != $field_name } { continue }

		# if we weren't given a module, or if this field is a keyword for all modules, return true
		if { $module == "" || $field_options(modules) == {} } { return 1 }

		# otherwise, check that the module we were given is in the list, if so return true
		if { [lsearch $field_options(modules) $module]>=0 } { return 1 }

		# if not, the field is not a keyword field:
		return 0
	}

	 # if we didn't find the field, return false
	return 0
}

#----------------------------------------# SUPPORT PROCEDURES #--------------------------------------#

# delete ALL the keywords for the master account
proc delete_keywords {} {
	variable company_id
	variable module
	variable number

	sql "
		delete from MASTER_KEYWORD 
		where COMPANY_ID = '$company_id'
		and MODULE = '$module'
		and NUMBER = $number
	"
}
# module - specific keyword procedures

proc product_keywords {} {
	global setup
	variable company_id
	variable module
	variable number

	set keywords {}

	# Enable keywords for searching for products on the website
	if { [setup product_keywords 0] } {
		set priority 3
		set contraction_priority [expr {$priority + 1}]

		foreach row [sql "select PHRASE from PART_PHRASE
		where COMPANY_ID = '$company_id' 
		and PART = $number"] {

			set phrase [lindex $row 0]
			set phrase_keywords [string_keywords $phrase]

			store_keywords $phrase_keywords $priority
		}
	}
}

proc customer_keywords {} {}
proc supplier_keywords {} {}
proc gl_keywords {} {}
proc asset_keywords {} {}

proc store_keywords { keywords priority } {
	variable master_keywords

	lappend_list master_keywords($priority) $keywords

	# find contractions, store them as lower priority
	set priority [expr { $priority + [setup keyword_contraction_priority_offset 1] }]
	if { $priority > 9 } { set priority 9 }
	
	foreach keyword $keywords {
		set contractions [contractions_of $keyword]
		lappend_list master_keywords($priority) $contractions
	}
}

proc is_ignore_word { word } {
	variable ignore_words

	if { [array names ignore_words $word] != {} } { 
		return 1 
	}
	return 0
}

#- load a field's value from a database table using master array
#- currently only supported if table is PRODUCT_EXTRAS
proc load_field_value { table_name field_name } {
	variable master
	
	set key {}
	array set key_data {}
	
	switch -- $table_name {
		PRODUCT_EXTRAS - PRODUCT_BARCODES {
			set key {COMPANY_ID PART}
			set key_data(COMPANY_ID) $master(COMPANY_ID)
			set key_data(PART) $master(NUMBER)
		}
		default {
			error "Unsupported table - \"$table_name\""
		}
	}
	
	# maybe change to retrieve all columns for some tables...
	set field_value ""
	query::find extra -table $table_name -key $key -key_data key_data \
			-fetch [list $field_name] -action {
		set field_value $extra($field_name)
	}
		
	return $field_value
}

#----------------------------------------# MISC #-----------------------------------------#

# add the items in list2 to list1 ($list1_name in calling procedure)
# same effect as concat, but should be more efficient if list1 is long and list2 not so much (see tcl doc. for lappend)
# maybe move to support/list.tcl .....
proc lappend_list { list1_name list2 } {
		upvar $list1_name list1
		eval "lappend list1 $list2"
}

#------------------------------------# STATIC NAMESPACE VARIABLES #---------------------------------#

variable last_sql_statement ""
variable minimum_word_length 2
variable maximum_word_length 14
variable minimum_contraction 3
variable maximum_contraction 14
variable separators " \t\n\"\*\/\\\$':#%!@&~+-,.;()\[\]{}<>?=_"
variable ignore_words 
foreach word { 	ago all am an and any are as at be by did do for get go got had
                has he if in is it its like me my no of oh on one only or our out
                per so the to too with 
} {
	set ignore_words($word) 1
}

# list of MASTER fields from which to derive keywords; Each item in list is {field search_priority ?module_list?}
# if there is no module list, the field applies to ALL modules
# if any changes are made to this, rebuild_keywords proc has to be called.
# may be overridden by setup option master_keyword_fields
variable master_keyword_fields
if { [setup master_keyword_fields {}] != {} } {
	set master_keyword_fields [setup master_keyword_fields]
} else {
	set master_keyword_fields {
		{NAME -priority 5}
		{PART -priority 1}
		{CONTACT_NAME -priority 5 -modules {AP AR}}
		{ADDRESS -priority 7 -modules {AP AR}}
		{EXTRA_DETAIL -priority 5 -ignore_words 1}
		{SERIAL_NUMBER -priority 5 -modules {FA}}
		{LOCATION -priority 5 -modules {FA}}
		{EMAIL_ADDRESS -priority 7 -modules {AP AR}}
		{SUPPLIER_PART_NO -priority 5 -modules {IN}}
		{SALES_MESSAGE -priority 5 -modules {FA IN} -ignore_words 1}
	}
	if { [::setup show_in_name2] != "" } {
		lappend master_keyword_fields {NAME2 -priority 5 -modules {IN}}
	}
	if { [::setup database_type] == "wct" } {
		lappend master_keyword_fields {SUPPLIER_PART_NO -priority 5 -modules {AP}}
	}
}

# end namespace:
}
