
# fbclient remote access to fbserver (client side API)
# Replaces the various copies of remote_connect etc

namespace eval fbremote {

	variable sockID ""
	variable server_name ""
	variable remote_state ""
	variable remote_length 0
	variable version2 0

	variable disconnected 1
	variable requestID 0
	variable responseID 0
	variable status
	variable data

# Establish socket connection with remote fbserver
# Call read_socket when socket is readable
# Send LOGIN command to fbserver

proc connect {host user password database port} {
	global setup
	variable sockID
	variable server_name
	variable remote_state
	variable version2
	variable disconnected
	variable requestID
	variable responseID
	variable status
	variable data

	# implement ...
	if {! [info exists setup(connect_timeout)]} {
		set setup(connect_timeout) 20
	}
	if {! [info exists setup(remote_timeout)]} {
		set setup(remote_timeout) 300
	}
	set server_name "$host:$port"
	if {$sockID != ""} { fbremote::disconnect }

	# Establish connection
	if {[catch {set sockID [socket $host $port]} msg]} {
		set disconnected 1
		set msg [string map [list "couldn't open socket: " ""] $msg]
		error "Cannot connect to fbserver $server_name\n$msg"
	}

	fileevent $sockID readable [list ::fbremote::read_socket]
	fconfigure $sockID -buffering line -translation binary -blocking 1

	# Established socket connection
	set requestID 0
	set responseID 0
	set disconnected 0
	set remote_state "get_length"
	array unset status
	array unset data
	array set status {}
	array set data {}

	set loginData "FBASE,$user,$password,$database"

	# Only if in proxy mode ...
	# Send login data twice so fbserver recognises we are a version2 client
	append loginData ",FBASE,$user,$password,$database"

	set result [fbremote::command "LOGIN" $loginData]
	set remoteStatus [lindex $result 0]
	set remoteData [lindex $result 1]

	if {$remoteStatus != "OK"} {
		fbremote::disconnect
		error "LOGIN to fbserver $server_name failed:\n$remoteData"
	}

	# Connected successfully
	# Save the remote companyID, and fbserver version
	if {[llength $remoteData] > 1} {
		set version2 1
		set setup(remote_company) [lindex $remoteData 0]
	} else {
		# Old fbserver
		set version2 0
		set setup(remote_company) $remoteData
	}
}

# Read data from socket
# Check for connection closed
# Check for connection timeout
# Get length, status and data
proc read_socket {} {
	global setup
	variable sockID
	variable server_name
	variable remote_state
	variable remote_length
	variable requestID
	variable responseID
	variable status
	variable data

	# If the gets command returns a negative value then the connection has been broken
	if {[gets $sockID request] < 0} {
		fbremote::disconnect
		after cancel fbremote::timeout

		# Wakeup every command procedure
		while {$responseID < $requestID} {
			incr responseID
			set status($responseID) "DISC"
			set data($responseID) "Lost connection to $server_name"
		}
		return
	}

	switch -- $remote_state {
		"get_length" {
			# Causes fbclient to lose the connection...
			# catch { progress::refresh_local }

			set length $request
			if {[string is integer -strict $length]} {
				set remote_length $length
			} else {
				fbremote::timeout
			}
			set remote_state "get_status"
		}
		"get_status" {
			#...
			incr responseID
			set status($responseID) $request
			set data($responseID) [read $sockID $remote_length]

			# cancel the timeout function
			after cancel fbremote::timeout
			set remote_state "get_length"
		}
		default {
			fbremote::timeout
		}
	}
}

# Timed-out waiting for response from fbserver
proc timeout {} {
	global setup
	variable server_name
	variable remote_state
	variable disconnected
	variable requestID
	variable responseID
	variable status
	variable data

	after cancel fbremote::timeout

	if {! $disconnected} {
		# Wakeup every command procedure
		while {$responseID < $requestID} {
			incr responseID
			set status($responseID) "ERROR"
			set data($responseID) "No response from $server_name after $setup(remote_timeout) seconds"
		}
		fbremote::disconnect
	}
	set remote_state "get_length"
}

# Send command to fbserver
# Wait for response
# Raise error or return result
proc command { command value } {
	global setup
	variable sockID
	variable server_name
	variable disconnected
	variable requestID
	variable responseID
	variable status
	variable data

	if {$sockID == ""} {
		fbremote::disconnect
		set remoteStatus "DISC"
		set remoteData "No connection to server $server_name"
		return [list $remoteStatus $remoteData]
	}

	# if there is an outstanding request, wait for the response
	# needs to wait for the immediately preceding outstanding request, currently this is waiting for the oldest outstanding request...
	if { $responseID < $requestID } {
		fbwait ::fbremote::data($requestID)
	}

	# Allocate next request ID
	set ID [incr requestID]

	puts -nonewline $sockID "[string length $value]\n"
	puts -nonewline $sockID "$command\n"
	puts -nonewline $sockID $value
	flush $sockID

	after [expr {$setup(remote_timeout) * 1000}] fbremote::timeout

	# Now wait for the response
	# getting errors on Priscilla's PC (WEKA) -- $data($ID) throws "no such element" -- don't know why...
	# this fix (looping) shouldn't be necessary ...
	while {! [info exists data($ID)]} {
		fbwait ::fbremote::data($ID)
	}

	set remoteStatus $status($ID)
	set remoteData $data($ID)
	unset status($ID)
	unset data($ID)

	# Error from fbserver
	if {$remoteStatus != "OK" && $remoteStatus != "QUIT"} {
		if {! $disconnected} {
			set remoteStatus "ERROR"
			set remoteData "Remote Server $server_name Error\n$remoteData"
		}
	}

	return [list $remoteStatus $remoteData]
}

# Send sql command to fbserver, wait for response
# For fbserver v2 the response format is as for softsql
proc sql { command } {
	global setup
	
	# Flush and clear query cache for all tables
	if {[info command ::query::flush_cache] != ""} {
		query::flush_cache -clear
	}

	# For fbserver v1 - special case for numrows
	if {$command == "numrows"} {
		if {$::fbremote::version2} { error "sql numrows not supported" }
		return $setup(remote_numrows)
	}

	# Sql command - access server and return result
	set result [fbremote::command "sql" $command]
	set remoteStatus [lindex $result 0]
	set remoteData [lindex $result 1]

	if {$remoteStatus != "OK"} {
		error $remoteData
	}
	if {$::fbremote::version2} { return $remoteData }

	set numrows [lindex $remoteData 0]
	set value [lindex $remoteData 1]
	set setup(remote_numrows) $numrows
	return $value
}

# Send sql command to fbserver, wait for response
# Response is list of list-form rows (as for softsql)
# Don't flush or clear the cache
proc sql_list { command } {
	# fbserver v2 doesn't support numrows
	if {$command == "numrows"} {
		error "sql numrows not supported"
	}

	# Sql command - access server and return result
	set result [fbremote::command "sql" $command]
	set remoteStatus [lindex $result 0]
	set remoteData [lindex $result 1]

	if {$remoteStatus != "OK"} {
		error $remoteData
	}
	return $remoteData
}

# Send sql command to fbserver, wait for response
# Response format is a list of array-form rows
# Don't flush or clear the cache
proc sql_array { command context } {
	set data [list $command $context]
	set result [fbremote::command "sql_array" $data]
	set remoteStatus [lindex $result 0]
	set remoteData [lindex $result 1]

	if {$remoteStatus != "OK"} {
		error $remoteData
	}
	return $remoteData
}

# Close socket connection
# Used by Consolidated_Profit_Loss.tcl etc
proc disconnect {} {
	variable sockID
	variable disconnected

	if {$sockID != ""} {
		if {[info command ::query::flush_cache] != ""} {
			query::flush_cache
		}
		catch { close $sockID }
		set sockID ""
	}
	set disconnected 1
}

}
