
# Emulation layer for fbsql.so/.dll
# Implements the sql command and subcommands
# Part of the fbproxy project - replacement for fbsql.so and fbsql.dll

# sql "select ..." returns the result in list-of-rows form, exactly as previously
# sql "insert/update ..." returns [list affected_rows insert_id]

# Single connection to fbserver
namespace eval sql {
	set fbproxy 1
	set connID ""
}

#---------------------------------------------------------------------------

# The sql procedure previously provided by fbsql.so/.dll
# Perform database query/statement
# Send request, wait for response (in list form)
# For an OK response, result is one row { affected_rows insert_id }

proc sql { command args } {
	global computer_name computer_user

	set context [get_calling_context "::sql"]
	switch -- $command {
		connect {
			if {[sql_connected]} { error "sql connect: already connected" }
			set proxy_host [lindex $args 0]
			set user [lindex $args 1]
			set password [lindex $args 2]
			set database [lindex $args 3]
			set proxy_port [lindex $args 4]

			# Delete when environment.tcl finished...
			if {![info exists computer_name]} { set computer_name "" }
			if {![info exists computer_user]} { set computer_user "" }

			set today [clock format [clock seconds] -format "%Y-%m-%d"]
			set result [proxy::connectdb $proxy_host $user $password $database $proxy_port \
								$computer_name $computer_user $today]
			set ::sql::connID [lindex $result 0]
			return [lindex $result 1]
		}
		selectdb {
			set database [lindex $args 0]
			if {$database == ""} { error "sql selectdb: no database specified" }
			if {![sql_connected]} { error "sql selectdb: not connected" }
			proxy::selectdb $::sql::connID $database
			return
		}
		query {
			set statement [lindex $args 0]
		}
		numrows { error "sql numrows: not supported" }
		disconnect {
			# Maybe set connID after disconnect ...
			if {![sql_connected]} { error "sql disconnect: not connected" }
			if {[info command ::query::flush_cache] != ""} {
				query::flush_cache
			}
			set ::sql::connID ""
			proxy::disconnect $::sql::connID
			return
		}
		version { return "Fastbase MySQL interface for Tcl: version 2.0" }
		startquery { error "sql startquery: not supported" }
		fetchrow { error "sql fetchrow: not supported" }
		endquery { error "sql endquery: not supported" }
		default { set statement $command }
	}

	# Query
	if {$statement == ""} { error "sql query: no query specified" }
	if {![sql_connected]} { error "sql query: not connected" }

	# Flush and clear query cache for all tables
	if {[info command ::query::flush_cache] != ""} {
		query::flush_cache -clear
	}
	return [sql_list_query $::sql::connID $statement $context]
}

# Return true if we are connected to fbproxy
proc sql_connected { } {
	return [expr {$::sql::connID != ""}]
}

# Query subcommand
proc sql_list_query { connID statement context } {
	# Refresh local progress bar; foreground jobs only
	if {$::event::limbo} { ::progress::refresh_local }

	# Send query request
	set id [proxy::sendListQuery $connID $statement $context]
	set result {}
	set final 1

	# Responses come back in blocks
	while true {
		set response [proxy::receive $connID $id]
		set type [lindex $response 0]
		switch -- $type {
			"ok" {
				set affected_rows [lindex $response 1]
				set insert_id [lindex $response 2]
				set result [list $affected_rows $insert_id]
				break
			}
			"error" {
				set reasonCode [lindex $response 1]
				set reasonString [lindex $response 2]
				proxy::requestComplete $connID $id $final
				error "sql error: $reasonCode $reasonString"
			}
			"list" {}
			default {
				proxy::requestComplete $connID $id $final
				error "sql error: list, ok or error expected"
			}
		}

		# Append contents of block to result
		set final [lindex $response 1]
		set block [lindex $response 2]
		eval lappend result $block
		if {$final} { break }
		if {$::event::limbo} { ::progress::refresh_local 100 }
	}
	proxy::requestComplete $connID $id $final
	return $result
}

# Perform array-form-result sql query
# Used by fbserver, to support query::forevery via fbclient
# Return result as a list of rows in array form
proc sql_array_query { connID statement context } {
	# Send query request
	set id [proxy::sendArrayQuery $connID $statement $context]
	set result {}
	set final 1

	# Responses come back in blocks
	while true {
		set response [proxy::receive $connID $id]
		set type [lindex $response 0]
		switch -- $type {
			"ok" {
				proxy::requestComplete $connID $id $final
				error "unexpected ok from proxy"
			}
			"error" {
				set reasonCode [lindex $response 1]
				set reasonString [lindex $response 2]
				proxy::requestComplete $connID $id $final
				error "sql error: $reasonCode $reasonString"
			}
			"array" {}
			default {
				proxy::requestComplete $connID $id $final
				error "sql error: array or error expected"
			}
		}
		set final [lindex $response 1]
		set block [lindex $response 2]
		eval lappend result $block
		if {$final} { break }
	}
	proxy::requestComplete $connID $id $final
	return $result
}

# Wait in the event loop until variable is set or unset
# original fbwait only allows file and timer events while waiting - not window or idle
proc fbwait { name } {
	# Or tkwait var $name ?
	vwait $name
}

# Determine callers context, the call-stack
# Used for passing to fbproxy to attach to query fingerprints
proc get_calling_context { context } {
	set level [info level]
	if {$level > 2} {
		set relative [lindex [info level -2] 0]
		if {$relative != ""} {
			set absolute [uplevel 2 namespace which -command $relative]
			set context "$absolute $context"
		}
	}
	if {$level > 3} {
		set relative [lindex [info level -3] 0]
		if {$relative != ""} {
			set absolute [uplevel 3 namespace which -command $relative]
			set context "$absolute $context"
		}
	}
	return $context
}

# -------------------------------------------------------------------------
# Refresh progress bar upon sql activity

# Create placeholder event variables 
if {! [namespace exists event]} {
	namespace eval event {
		variable limbo 0
	}
}

# Create placeholder progress procs
if {! [namespace exists progress]} {
	namespace eval progress {
		proc refresh_local { {percent 50} } {}
		proc finish_local { } {}
	}
}

# Create fastbase_office variable
if { ![info exists fastbase_office] } {
	set fastbase_office 0
}
