
# environment.tcl  Standard runtime environment and network settings
# All scripts and gui programs should use this at start-up

namespace eval environment {

# -------------------------------------------------------------------------

# check tcl version - min 8.4, 8.5 for websites, 8.6 for ttk ...
# is tk or ttk loaded ...
# return error, or requires error handling or popup ...

# Initialise globals and setup variables, connect to database
# Seperate init for gui, script, website ...
proc init_script { home_directory } {
	global argv

	standard_libraries $home_directory
	standard_globals

	# Parse command arguments and options
	command::arguments $argv opt {database} {database "" setup ""}
	if {$opt(database) != ""} {
		setup database_name $opt(database)
	}
	standard_setup_files $opt(setup)
	connect_database
	setup_defaults
}

# Load standard source files for Fastbase libraries, and/or establish auto load
# Assumes all standard Tcl & Tk libraries are already loaded
proc standard_libraries { home_directory } {
	variable libraries_initialised
	global auto_path

	# Once per process (don't repeat if run from fbclient or fbserver)
	if {[info exists libraries_initialised]} { return }

	# Ensure correct setup proc is loaded; extra copies exist
	# Move toolkit/setup.tcl to common/setup.tcl ...
	set fileName $home_directory/toolkit/setup.tcl
	if {[file exists $fileName]} {
		namespace eval :: [list source $fileName]
	}

	# Load standard database schema
	set fileName $home_directory/common/schema_table.tcl
	if {[file exists $fileName]} {
		namespace eval :: [list source $fileName]
	}
	set fileName $home_directory/common/schema_column.tcl
	if {[file exists $fileName]} {
		namespace eval :: [list source $fileName]
	}

	# Load redis interface
	set fileName $home_directory/common/redis/redis.tcl
	if {[file exists $fileName]} {
		namespace eval :: [list source $fileName]
	}

	# Load address library
	set fileName $home_directory/common/address/country.tcl
	if {[file exists $fileName]} {
		namespace eval :: [list source $fileName]
	}

	# Source a standard logging proc ...

	# List of well-known directories for libraries
	# The package facility searches these directories (and their sub-directories) for packages
	# Unknown commands search these directories also, see auto_mkindex
	lappend auto_path $home_directory
	lappend auto_path $home_directory/support
	lappend auto_path $home_directory/common
	lappend auto_path $home_directory/toolkit

	# Add environment::source procedure, that works like source, with setup-like caching...

	set libraries_initialised 1
}

# Set standard Fastbase global variables
# This may be called either before or after standard_libraries
# Standard folder & file locations, linux or windows, gui or cmd ...
# Not here - create initial namespaces, login etc ...
proc standard_globals { } {
	global env
	global tcl_version
	global tk_version
	global gui_script
	global windows64
	global computer_name
	global computer_user
	global fastbase_office
	global user_login_date
	global user_login_time
	global unique
	global history
	variable globals_initialised

	# Once per process
	if {[info exists globals_initialised]} { return }

	# Should be 8.4, 8.5 or 8.6
	set tcl_version [string range [info patchlevel] 0 2]

	# Exists if running wish; 8.4 or 8.6
	set gui_script [info exists tk_version]

	# Windows 64 bit
	set windows64 0
	if {[info exists env("ProgramFiles(x86)")]} {
		set windows64 1
	}

	if {[catch { set computer_name $env(COMPUTERNAME) }]} {
		set computer_name [info hostname]
	}
	if {[catch { set computer_user $env(USERNAME) }]} {
		set computer_user ""
		catch { set computer_user $env(USER) }
	}
	set fastbase_office 0
	if {$computer_name == "STARLING" || $computer_name == "WEKA" || $computer_name == "BEETLE" ||
		$computer_name == "MAGPIE" || $computer_name == "HUIA" || $computer_name == "SPARROW" ||
		$computer_name == "moa" || $computer_name == "FINCH" || $computer_name == "VULTURE"} {
		set fastbase_office 1
	}

	set user_login_date ""
	set user_login_time ""

	# Unique window identifier
	set unique 0
	# Clear the history field in case it doesn't get set (see login::exit)
	set history {}

	# Check that env(TEMP) exists, it won't on unix based systems
	if {[catch { set env(TEMP) }]} {
		set env(TEMP) "/tmp"
	}

	# Create dummy winfo procedure for setup.dat
	# Should have a Fastbase wrapper for winfo instead ...
	if {[info commands ::winfo] == ""} {
		proc ::winfo { args } { }
	}
	set globals_initialised 1
}

# Read the standard setup files: setup.dat, fastbase.dat etc
# Assumes current directory is correct
proc standard_setup_files { {extraFile ""} } {
	variable setup_files_initialised
	global argv0

	# Once per process
	if {[info exists setup_files_initialised]} { return }

	# setup.dat is usually present
	if {[file exists setup.dat]} {
		setup::useFile setup.dat
	}

	# From as.tcl
	if {[file exists local.dat]} {
		setup::useFile local.dat
	}

	# Probably from a command-line argument
	if {$extraFile != ""} {
		setup::useFile $extraFile
	}

	# Only if script is fbclient.tcl
	set script [string tolower [file tail $argv0]]
	if {$script == "fbclient.tcl" && [file exists "../fastbase.dat"] } {
		setup::useFile ../fastbase.dat
	}

	# Load the c:/fastbase.dat file also (originally from as.tcl for fbclient only)
	if {$script == "fbclient.tcl" && [file exists "c:/fastbase.dat"]} {
		setup::useFile c:/fastbase.dat
	}

	# Should have loaded at least one setup file
	# Popup message for as.tcl, fbclient or fbserver...
	if {[llength $setup::nameList] == 0} {
		error "cannot open setup file (setup.dat)"
	}
	set setup_files_initialised 1
}

# Initialise sql namespace, connect to the database
# Single connect only; doesn't re-connect (to other database)
# Doesn't return result of connect, or catch error
proc connect_database { } {
	variable database_initialised
	variable database_name
	global argv

	# Once per process, and after sql::disconnect
	if {[info exists database_initialised]} {
		if {$database_initialised} { return }
	}

	# Current database, for query procs
	set database_name [setup database_name]

	# Connect to database
	# Hide argv from sql.tcl (delete this)...
	set argv {}
	namespace eval :: { source sql.tcl }
	sql::connect
	sql::setup_tables

	# For query/cache
	set ::property(connID) $::sql::connID

	# Standard database schema ...
	set database_initialised 1

	# Connect to redis server (if not already connected)
	if {[setup redis_server] != "" && ![info exists ::property(redisID)]} {
		set redisID [redis::connect [setup redis_server]]

		# New proc command:set_properties ...
		set ::property(redisID) $redisID
		set ::property(domain) [setup database_name]
		# create redis session ...
	}
}

# Disconnect from database and from redis
proc disconnect { } {
	variable database_initialised

	if {[info exists database_initialised] && $database_initialised} {
		sql::disconnect
	}

	# Disconnect from redis server
	if {[info exists ::property(redisID)]} {
		set redisID [command::inherit property {redisID}]
		catch { redis::flush_all $redisID }
		catch { redis::disconnect $redisID }
		# Also disconnect subscribe connection...

		unset ::property(redisID)
	}
}

# Determine primary keys etc for standard database tables ...
proc standard_tables { } {
}

# Defaults for setup options
# Only for widely-used setup options, most defaults are given at point-of-use
# These should eventually be moved to suitable init procs
proc setup_defaults { } {
	variable defaults_initialised
	global env

	# Once per process
	if {[info exists defaults_initialised]} { return }

	# Connect to the database via fbproxy
	setup fbproxy 0
	# Proxy server hostname
	setup proxy_host "127.0.0.1"

	# database_name from argv ...
	# database_name should always be lowercase ?

	# set the database type after connecting, sql::connect may use command line
	# database_type should always be lowercase ...
	setup database_type [setup database_name]

	# Copy all order lines to invoice even if zero quantity; default true for BE
	# Should be in BE setup.dat ...
	if {[setup database_type] == "briteuro"} {
		setup order_invoice_all 1
	}

	# Used only in AKB stationary layouts (re printing images upside down)
	setup akbear_normal 1

	# Default location FastBase will prompt to save files (eg: reports saved to disk)
	setup default_directory {c:/My Documents}

	# Location of the photos folder (must be accessible from all computers that want to see photos)
	setup image_directory "../photos"

	# Image filename for company logo to display on the main screen, background #f5f5f5
	setup company_logo "images/[setup database_name]-logo.gif"

	# The smtp host server hostname and port, for sending emails
	# The authentication username and password are not usually required
	setup smtphost ""
	setup smtpport "25"
	setup smtp_username ""
	setup smtp_password ""

	# Email address for notifications (not errors)
	setup email_notify "notify@fastbase.co.nz"

	# Default email address to send support emails to
	setup email_support "support@fastbase.co.nz"

	# Customer price books or discount expiry notices are sent from this address
	setup email_sales [setup email_support]

	# The fbserver hostname if we're using one, otherwise empty string
	setup remote_host ""
	setup remote_company ""

	# Number of background jobs running; shouldn't be in setup ...
	setup background_jobs 0

	# Maximum length of account numbers (on screen and when printing)
	# Changing this would require changing length of database columns also
	setup length_of_account 30

	# Name of the field for Prompt Payment Discounts
	# McPhail Group use this field and call it 'Late Payment Fee'
	# Code tests on this default value; should use separate setup vars for title & true/false ...
	setup prompt_payment "Prompt Payment Discount"

	# True if a Hohepa Homes company
	setup hohepa_homes 0

	# Number of decimal places used (on-screen and printed)
	setup stock_price_decimals 4
	setup stock_cost_decimals 4
	setup stock_qty_decimals 2

	# 21 exclamation marks (an impossible part number)
	setup buildmisc_part "!!!!!!!!!!!!!!!!!!!!!"

	# List of all 'Web Display' options
	setup web_display_options {{N "No"} {Y "Yes"} {S "Special"} {F "Flier"}}

	# setup oe_priority {{E "Enquiry"} {B "Breakdown"} {U "Urgent"} {Q "Quote for Contract"} {S "Stock"}}
	setup oe_priority {{N "Normal"} {U "Urgent"} {C "Critical"}}

	# Name of the first bin location field
	setup bin_location1 "Bin Location"

	# Name of the profit column (in reports)
	setup profit_column "Profit"

	# Run export manifest overnight
	setup overnight 0

	# Screen viewer; should be namespace vars ...
	setup match_case 0
	setup find_history {}
	setup find_string ""

	setup frame_stack {}

	# Things that shouldn't be in the setup array:
	#   frame_stack
	#   markup_tables
	#   background_printing
	#   background_jobs
	#   inline_fax_printer (see output.tcl)

	set defaults_initialised 1
}

# End namespace
}

