
# command.tcl  Command line arguments, calling-convention helper procs 

# Use command::arguments both for script command lines and internal commands
# Any proc with more than about three arguments should use this
# Doesn't support multiple of the same option
# The order of positional arguments (that aren't options) is preserved
# The order of options is not preserved

namespace eval command {

# -------------------------------------------------------------------------

# Set options in opts array (without dash) and apply default values
#   commandArgs - arguments passed to the command
#   optsName - options array, options and arguments are placed in here
#   argumentList - names of positional arguments (required unless default provided)
#   defaultList - {name value ... } default values
# For an option name without a value, assume value is 1 (true)
proc arguments { commandArgs optsName argumentList {defaultList {}} } {
	upvar $optsName opts

	# Parse list of options and arguments, populate opts
	set argNo 0
	set use_previous 0
	foreach arg $commandArgs {
		if {$use_previous} {
			if {[string match {-[A-Za-z]*} $previous]} {
				# -optionname optionvalue
				set name [string range $previous 1 end]
				if {[string match {-[A-Za-z]*} $arg]} {
					# Arg is the next -optionname
					set value 1
				} else {
					set opts($name) $arg
					set use_previous 0
					continue

				}
			} else {
				# Un-named argument
				set value $previous
				set name [lindex $argumentList $argNo]
				# Extra positional argument
				if {$name == ""} { error "unexpected extra argument: $value" }
				incr argNo
			}
			set opts($name) $value
		}
		set previous $arg
		set use_previous 1
	}

	# Last option or argument
	if {$use_previous} {
		if {[string match {-[A-Za-z]*} $previous]} {
			set name [string range $previous 1 end]
			set value 1
			set opts($name) $value
		} else {
			# Un-named last argument
			set value $previous
			set name [lindex $argumentList $argNo]
			if {$name == ""} { error "unexpected last argument: $value" }
			set opts($name) $value
		}
	}

	# Apply default values for options and arguments
	apply_defaults opts $defaultList

	# Check for missing arguments
	# Include proc name of caller ...
	foreach name $argumentList {
		if {! [info exists opts($name)]} { error "missing argument: \"$name\" args \"$commandArgs\"" }
	}
}

# Copy properties from ancestor procs to parent proc's property array
# Return the value of the last property in the list
# Typically used to get an implicit parameter passed to a procedure (once only)
proc inherit { propertyName propertyList } {
	upvar $propertyName property

	# Find ancestor procs with a property array
	set index 0
	set lowest [expr {[info level] -2}]
	for {set level $lowest} {$level >= 0} {incr level -1} {
		upvar #$level $propertyName ancestor$index
		if {[array exists ancestor$index]} { incr index }
	}
	set count $index
	set value ""

	# Each property to find
	foreach name $propertyList {
		# Property is already in parent
		if {[info exists property($name)]} {
			set value $property($name)
			continue
		}

		# Search ancestor procs for property
		for {set index 0} {$index < $count} {incr index} {
			if {[info exists ancestor${index}($name)]} {
				set value [set ancestor${index}($name)]
				set property($name) $value
				break
			}
		}
		# Not found
		if {$index == $count} { error "inherit: cannot find $name" }
	}
	return $value
}

# Copy all properties from ancestor procs to parents property array
# Overwrite any existing properties in parent
# Does nothing if no ancestor has a property array
proc inherit_all { propertyName } {
	upvar $propertyName property

	# Find ancestor procs with a property array
	set lowest [expr {[info level] -2}]
	for {set level 0} {$level <= $lowest} {incr level} {
		upvar #$level $propertyName ancestor
		if {[array exists ancestor]} {
			array set property [array get ancestor]
		}
	}
}

# Make a command list, for passing as a tk callback
# Pass current properties to the command
# ns is the required namespace (usually namespace of the caller)
proc make_callback { ns procName args } {
	command::inherit_all property
	set propertyList [array get property]
	return [list ::command::on_callback $ns $procName $args $propertyList]
}

# Called by event loop on event
# Set properties, invoke command in namespace with arguments
proc on_callback { ns procName argList propertyList } {
	array set property $propertyList
	namespace eval $ns [list $procName] $argList
}

# Apply default options (list in array get format)
proc apply_defaults { optsName defaultList } {
	upvar $optsName opts

	foreach {name value} $defaultList {
		if {! [info exists opts($name)]} {
			set opts($name) $value
		}
	}
}

# End namespace
}

