
# Database queries, using proxy and the cache
# Replacements for the sql namespace procedures

# Flush cache on timeout or ui idle ...
# Flush write cache on window loses focus ...
# Flush read cache on window gains focus ...

namespace eval query {

# Get zero or more rows, from cache or database
# Invoke action per row, return number of rows found
# Arguments and options:
#   row            Name of array where row-result returned
#   table          Database table name
#   key            List of column names forming the query key (the where clause)
#   key_data       Name of array with key columns (name and value)
#   <column>       Key-column name and value may be supplied as an option
#   fetch          List of column names to fetch from the database, default all
#   count          Return row count only, don't fetch row data (separate proc ...)
#   order          List of column name to order by ( ^column = descending)
#   min            Error if retrieved less than min rows
#   max            Error if retrieved greater than max rows (-1 = no maximum)
#   action         Invoke action on each row returned, allows break and continue commands
#   rows           Name of variable, list of rows each in list form (must use fetch)
#   nocache        If true don't use cache

proc find { args } {
	# Get arguments and options
	set defaults {table "" key_data "" fetch {} count 0 order {} min 0 max -1 action {} rows "" nocache 0}
	command::arguments $args opt {row table action} $defaults
	set connID [command::inherit property {connID}]
	global msecNow

	# Where row result returned
	upvar $opt(row) resultrow

	# Table name defaults to name of row array
	set table $opt(table)
	if {$table == ""} {
		set table [string toupper $opt(row)]
	}

	# Use default key from schema
	if {[info exists opt(key)]} {
		set key_columns $opt(key)
	} else {
		set key_columns $::schema::key_columns($table)
	}
	set key_columns [schema::get_sorted_key $table $key_columns]

	# Key column names and values, default to row array
	if {$opt(key_data) == ""} {
		upvar $opt(row) key_data
	} else {
		upvar $opt(key_data) key_data
	}

	# Fetch row count or fetch column data
	# Fetch all columns if fetch empty or not supplied
	set count_only $opt(count)
	set fetch_columns $opt(fetch)
	set fetch_all [expr {$count_only == 0 && $fetch_columns == ""}]

	set order $opt(order)
	set min_rows $opt(min)
	set max_rows $opt(max)
	set action $opt(action)

	# Set list of rows, each row in list form
	if {$opt(rows) == ""} {
		set listForm 0
	} else {
		set listForm 1
		upvar $opt(rows) rowslist
	}

	# Bypass cache
	set nocache $opt(nocache)

	# Determine query key (used as cache index)
	set key_value [make_query_key $table $key_columns opt key_data]

	# -------------------------------------------------------
	# Check cache

	# if nocache just do the query ...
	
	set cache_ns [make_cache_ns $connID]
	set table_ns [make_table_ns $cache_ns $table]
	set table_key [get_table_key $table_ns $key_columns]

	# Find existing cache entry
	set found 0
	set missing 0
	set dirtyFlag 0
	if {[info exists ${table_key}($key_value)]} {
		set cache_entry [set ${table_key}($key_value)]
		set dirtyFlag [lindex $cache_entry 0]
		set dirtyList [lindex $cache_entry 1]
		set intoTime [lindex $cache_entry 2]
		set allColumns [lindex $cache_entry 3]
		set orderBy [lindex $cache_entry 4]
		set rowCount [lindex $cache_entry 5]
		set rowData [lindex $cache_entry 6]
		
		set msecNow [clock clicks -milliseconds]
		if {[alive $cache_ns $table $intoTime]} {
			# OrderBy matches, or less than 2 rows
			if {$order == "" || $order == $orderBy || $rowCount < 2} {
				set found 1

				# Columns don't matter if there's zero rows
				if {$rowCount > 0 && !$allColumns} {
					if {$fetch_all} {
						set missing 1
					} else {
						# Are all fetch_columns in cached row(s)
						array set first_row [lindex $rowData 0]
						foreach name $fetch_columns {
							if {![info exists first_row($name)]} {
								# Required column is missing
								set first_row($name) ""
								set missing 1
							}
						}
					}
				}
			}
		}
	}

	# -------------------------------------------------------
	# Flush dirty cache, build and perform query, update cache

	if {!$found || $missing} {
		set context [get_calling_context "::query::find"]
		if {$dirtyFlag} {
			# Found cache entry but order doesn't match or there's missing columns
			flush_cache_entry $table $context $table_key $key_columns $key_value
		}
		if {!$found} {
			# Flush matching other-keys for this table
			foreach other_columns [all_table_keys $table_ns] {
				if {$other_columns == $key_columns} { continue }
				set pattern [make_key_pattern $key_columns $other_columns $key_value]
				set matchList [get_matching_entries $table_ns $other_columns $pattern]
				
				set other_key [get_table_key $table_ns $other_columns]			
				foreach name $matchList {
					flush_cache_entry $table $context $other_key $other_columns $name
				}
			}
		}

		# Build fetch clause
		if {$fetch_all} {
			set fetch_clause "*"
		} else {
			if {$count_only} {
				# Retrieve row count only
				set fetch_clause "count(*)"
			} elseif {$missing} {
				# Found the cache entry, but it has missing columns
				set fetch_clause [join [lsort [array names first_row]] ","]
			} else {
				# Fetch just the requested columns
				set fetch_clause [join $fetch_columns ","]
			}
		}

		# Build select query
		set where_clause [make_where_clause $table $key_columns $key_value]
		set order_clause [make_order_clause $order]
		set statement "select ${fetch_clause} from ${table}${where_clause}${order_clause}"

		if {$where_clause == "none"} {
			# Query would return zero rows
			set rowData {}
			set rowCount 0
		} else {
			# Perform query either via fbserver or fbproxy
			if {[setup remote_host] != ""} {
				# Connected via fbclient
				# Offer reconnect ...
				set rowData [fbremote::sql_array $statement $context]
			} else {
				# Connected via fbproxy
				set connID $property(connID)
				if {$connID == ""} { error "find: not connected" }
				set rowData [sql_array_query $connID $statement $context]
			}
			# Counted rows, no row data to cache
			if {$count_only} {
				set rowCount [lindex [lindex $rowData 0] 1]
				if {$rowCount == ""} { error "find: failed to get row count" }
				set rowData {}
			} else {
				set rowCount [llength $rowData]
			}
		}

		if {$rowCount < 5000 || $count_only} {
			# Update cache with query response
			set dirtyFlag 0
			set dirtyList {}
			set msecNow [clock clicks -milliseconds]
			set intoTime $msecNow
			set allColumns $fetch_all
			set cache_entry [list $dirtyFlag $dirtyList $intoTime $allColumns $order $rowCount $rowData]
			set ${table_key}($key_value) $cache_entry
		} else {
			# Don't store large results in cache 
			unset -nocomplain -- ${table_key}($key_value)
		}
	}

	# -------------------------------------------------------
	# Return result to caller

	# Throw error if rowCount < min or > max
	if {$rowCount < $min_rows} {
		error "find: got $rowCount row(s) minimum $min_rows"
	}
	if {$max_rows >= 0 && $rowCount > $max_rows} {
		error "find: got $rowCount row(s) maximum $max_rows"
	}

	# Invoke action on each row
	# Row only includes the changed fields
	if {$listForm} { set rowslist {} }
	if {$count_only} { set rowData {} }
	array set rowArray {}

	foreach row $rowData {
		array set rowArray $row
		if {$fetch_all} {
			# Copy complete array to caller
			array set resultrow [array get rowArray]
		} else {
			# Only copy the fetch columns
			foreach name $fetch_columns {
				set resultrow($name) $rowArray($name)
			}
		}

		# Pass row to caller in list form
		if {$listForm} {
			set valueList {}
			foreach name $fetch_columns {
				lappend valueList $rowArray($name)
			}
			lappend rowslist $valueList
		}

		# 0 = ok, 1 = error, 2 = return, 3 = break, 4 = continue
		if {$action != ""} {
			set code [catch { uplevel 1 $action } msg]
			switch -- $code {
				0 { }
				1 { return -code error \
							-errorinfo $::errorInfo \
							-errorcode $::errorCode $msg }
				2 { return -code return $msg }
				3 { return {} }
				4 { }
				default { return -code $code $msg }
			}
		}
	}
	return $rowCount
}

# Insert row, into cache or database
# Build sql query string
# Return insert_id (for auto increment table)
# May raise error on duplicate row
# Arguments and options:
#   row            Name of array with columns to insert
#   table          Database table name
#   insert_columns List of column names to insert (default all)
#   <column>       Insert-column name and value may be supplied as an option
#   nocache        If true don't use cache

proc insert { args } {
	# Get arguments and options
	set defaults {table "" insert_columns {} nocache 0}
	command::arguments $args opt {row table} $defaults
	set connID [command::inherit property connID]
	global msecNow

	# Where row to insert is supplied
	upvar $opt(row) insertrow

	# Table name defaults to name of row array
	set table $opt(table)
	if {$table == ""} {
		set table [string toupper $opt(row)]
	}

	# Default to all columns, from schema
	set insert_columns $opt(insert_columns)
	set allColumns 0
	if {$insert_columns == ""} {
		set insert_columns $::schema::all_columns($table)
		set allColumns 1
	} else {
		set insert_columns [schema::get_sorted_key $table $insert_columns]
	}

	# Bypass cache
	set nocache $opt(nocache)

	# Auto increment column name
	set auto_column $::schema::auto_column($table)
	if {$auto_column != ""} {
		set opt($auto_column) 0
	}

	# Merge values supplied as options and insert_columns
	array set merged {}
	foreach name $insert_columns {
		if {[info exists opt($name)]} {
			set value $opt($name)
		} else {
			if {[catch {set value $insertrow($name) }]} { error "insert column $name missing" }
		}
		set merged($name) [canonicalise $table $name $value]
	}

	# -------------------------------------------------------
	# Check cache

	# Flush & clear all keys except unique keys for this table
	set cache_ns [make_cache_ns $connID]
	set table_ns [make_table_ns $cache_ns $table]
	set context [get_calling_context "::query::insert"]
	
	# Find all matching cache entries, except unique keys
	foreach key_columns [all_table_keys $table_ns] {
		# Ignore unique keys, these will be handled explicitly
		if {[schema::is_unique_key $table $key_columns]} { continue }
		# Build key_value
		set key_value {}
		foreach name $key_columns {
			lappend key_value $merged($name)
		}
		set pattern [make_key_pattern $insert_columns $key_columns $key_value]
		set matchList [get_matching_entries $table_ns $key_columns $pattern]

		# Instead, check each entry for duplicates ...
		foreach name $matchList {
			set table_key [get_table_key $table_ns $key_columns]
			clear_cache_entry $table $context $table_key $key_columns $name
		}
	}

	# if nocache just do the query ...

	set found_all 0
	set dirtyFlag 0
	if {$auto_column == ""} {
		# Find cache entry per unique key for the table
		set found_all 1
		foreach uniqueKey $::schema::key_unique($table) {
			set key_columns [schema::get_sorted_key $table $uniqueKey]
			set table_key [get_table_key $table_ns $key_columns]
			set key_value {}
			foreach name $key_columns {
				lappend key_value $merged($name)
			}
			# Find existing cache entry
			if {[info exists ${table_key}($key_value)]} {
				set cache_entry [set ${table_key}($key_value)]
				set dirtyFlag [lindex $cache_entry 0]
				set dirtyList [lindex $cache_entry 1]
				set intoTime [lindex $cache_entry 2]
				set rowCount [lindex $cache_entry 5]
				set rowData [lindex $cache_entry 6]

				if {$dirtyFlag} {
					flush_cache_entry $table $context $table_key $key_columns $key_value
				}
				set msecNow [clock clicks -milliseconds]
				if {[alive $cache_ns $table $intoTime]} {
					if {$rowCount > 0} {
						# Raise error for duplicate row (include where in the error)
						set where [make_where_clause $table $key_columns $key_value]
						error "insert: duplicate row in cache for $table$where"
					}
				} else {
					# Entry has expired
					set found_all 0
				}
			} else {
				# Not found
				set found_all 0
			}
		}
	}

	# -------------------------------------------------------
	# Write to cache or database

	set insert_id 0
	if {$found_all} {
		# All of the existing unique cache entries are alive and zero rows
		set msecNow [clock clicks -milliseconds]
		set intoTime $msecNow
		# Cache is now dirty (un-written insert)
		set dirtyFlag 1
	} else {
		# Build and perform insert query, update cache
		set insert_id [perform_insert $table merged $context]
		set msecNow [clock clicks -milliseconds]
		set intoTime $msecNow

		# Put value of auto increment column in cache
		if {$auto_column != ""} {
			set merged($auto_column) $insert_id
			set insertrow($auto_column) $insert_id
		}
		# Cache is clean (agrees with the database)
		set dirtyFlag 0
	}

	# Update cache with single inserted row
	set dirtyList {}
	set order {}
	set rowData [list [array get merged]]
	set rowCount 1
	set cache_entry [list $dirtyFlag $dirtyList $intoTime $allColumns $order $rowCount $rowData]

	# Write to cache per each unique index
	foreach uniqueKey $::schema::key_unique($table) {
		set key_columns [schema::get_sorted_key $table $uniqueKey]
		set table_key [get_table_key $table_ns $key_columns]
		set key_value {}
		foreach name $key_columns {
			lappend key_value $merged($name)
		}
		set ${table_key}($key_value) $cache_entry

		# Only one cache entry needs to be dirty, for the first unique index
		# Otherwise the eventual insert would get a duplicate error
		if {$dirtyFlag} {
			set dirtyFlag 0
			set cache_entry [list $dirtyFlag $dirtyList $intoTime $allColumns $order $rowCount $rowData]
		}
	}
	return $insert_id
}

# Update zero or more rows, in cache or database
# Build sql query string
# Return number of rows affected
# Arguments and options:
#   row            Name of array with columns to update (the set columns)
#   table          Database table name
#   key            List of column names forming the query key (the where clause)
#   key_data       Name of array with key columns (name and value)
#   <column>       Key-column or set column may be supplied as an option
#   set            List of column names to update (default all)
#   min            Error if updated less than min rows
#   max            Error if updated greater than max rows (-1 = no maximum)
#   nocache        If true don't use cache

proc update { args } {
	# Get arguments and options
	set defaults {table "" key_data "" set {} min 0 max -1 nocache 0}
	command::arguments $args opt {row table} $defaults
	set connID [command::inherit property connID]
	global msecNow

	# Where row data to update is supplied
	upvar $opt(row) updaterow

	# Table name defaults to name of row array
	set table $opt(table)
	if {$table == ""} {
		set table [string toupper $opt(row)]
	}

	# Use default key from schema
	if {[info exists opt(key)]} {
		set key_columns $opt(key)
	} else {
		set key_columns $::schema::key_columns($table)
	}
	set key_columns [schema::get_sorted_key $table $key_columns]

	# Key column names and values, default to row array
	if {$opt(key_data) == ""} {
		upvar $opt(row) key_data
	} else {
		upvar $opt(key_data) key_data
	}

	# Columns to update
	set update_columns $opt(set)
	set allColumns 0
	if {$update_columns == ""} {
		# Default all columns except key_columns
		array set all_except {}
		foreach name $::schema::all_columns($table) {
			set all_except($name) 1
		}
		foreach name $key_columns {
			unset -nocomplain all_except($name)
		}
		set update_columns [array names all_except]
		set allColumns 1
	}

	set min_rows $opt(min)
	set max_rows $opt(max)

	# Bypass cache
	set nocache $opt(nocache)

	# Determine query key
	set key_value [make_query_key $table $key_columns opt key_data]

	# Merge values supplied in updaterow and options
	array set merged {}
	foreach name $update_columns {
		if {[info exists updaterow($name)]} {
			set value $updaterow($name)
		} else {
			if {[catch {set value $opt($name) }]} { error "update column $name missing" }
		}
		set merged($name) [canonicalise $table $name $value]
	}

	# Determine query key post-update (may be different)
	set new_key_value {}
	foreach name $key_columns value $key_value {
		if {[info exists merged($name)]} {
			lappend new_key_value $merged($name)
		} else {
			lappend new_key_value $value
		}
	}
	# Update modifies the query key
	set overlap [expr {$key_value != $new_key_value}]

	# -------------------------------------------------------
	# Check cache

	# if nocache just do the query ...
	
	set cache_ns [make_cache_ns $connID]
	set table_ns [make_table_ns $cache_ns $table]
	set table_key [get_table_key $table_ns $key_columns]
	set context [get_calling_context "::query::update"]
	
	# Flush and clear matching other-keys for this table
	foreach other_columns [all_table_keys $table_ns] {
		if {$other_columns == $key_columns} { continue }
		set pattern [make_key_pattern $key_columns $other_columns $key_value]
		set matchList [get_matching_entries $table_ns $other_columns $pattern]
				
		set other_key [get_table_key $table_ns $other_columns]			
		foreach name $matchList {
			clear_cache_entry $table $context $other_key $other_columns $name
		}
	}

	set found 0
	set dirtyFlag 0
	if {$overlap} {
		# Flush and clear cache entries for both old and new keys
		clear_cache_entry $table $context $table_key $key_columns $key_value
		clear_cache_entry $table $context $table_key $key_columns $new_key_value
	} else {
		# Find existing cache entry
		if {[info exists ${table_key}($key_value)]} {
			set cache_entry [set ${table_key}($key_value)]
			set dirtyFlag [lindex $cache_entry 0]
			set dirtyList [lindex $cache_entry 1]
			set intoTime [lindex $cache_entry 2]
			set allColumns [lindex $cache_entry 3]
			set orderBy [lindex $cache_entry 4]
			set rowCount [lindex $cache_entry 5]
			set rowData [lindex $cache_entry 6]
			
			set msecNow [clock clicks -milliseconds]
			if {[alive $cache_ns $table $intoTime]} {
				set found 1
				# Identify update columns that are and aren't already in the cache
				if {$rowCount > 0} {
					array set first_row [lindex $rowData 0]
					array set present {}
					array set absent {}
					foreach name $update_columns {
						if {[info exists first_row($name)]} {
							set present($name) 1
						} else {
							set absent($name) $merged($name)
						}
					}
				}
			}
		}
	}

	# -------------------------------------------------------
	# Write to cache or database

	if {$found} {
		# Leave cache un-touched, return zero rows affected
		if {$rowCount == 0} { return 0 }

		# Update cache, write to all rows in cache
		array set rowArray {}
		set newRowData {}
		set dirtyInsert [expr {$dirtyFlag && $dirtyList == ""}]

		array set dirtyArray {}
		foreach name $dirtyList {
			set dirtyArray($name) 1
		}
		# Merge absent columns into dirtyArray
		array set dirtyArray [array get absent]
		set presentNames [array names present]
		set absentColumns [array get absent]

		if {$rowData == ""} {
			# Cache has count only; repeat rowCount times
			array set dirtyArray $absentColumns
			for {set rowNo 0} {$rowNo < $rowCount} {incr rowNo} {
				lappend newRowData $absentColumns
			}
		} else {
			foreach row $rowData {
				array set rowArray $row
				# Merge absent values into row
				array set rowArray $absentColumns

				# Is present value different from update value
				foreach name $presentNames {
					if {$rowArray($name) != $merged($name)} {
						set rowArray($name) $merged($name)
						set dirtyArray($name) 1
					}
				}
				lappend newRowData [array get rowArray]
			}
		}

		# Turn dirty array back into list
		if {!$dirtyInsert} {
			set dirtyList [array names dirtyArray]
			if {[llength $dirtyList]} { set dirtyFlag 1 }
		}
		# May have updated columns that are part of unique keys
		set updated_unique 0
		foreach uniqueKey $::schema::key_unique($table) {
			foreach name $uniqueKey {
				if {[info exists dirtyArray($name)]} {
					set updated_unique 1
					break
				}
			}
		}
		# May have updated order-by columns
		set order {}
		if {$updated_unique} {
			# Need to do the insert or update now; may get duplicate error
			# Save the cache entry
			set cache_entry [list $dirtyFlag $dirtyList $intoTime $allColumns $order $rowCount $newRowData]
			set ${table_key}($key_value) $cache_entry

			# Flush insert or update to database
			flush_cache_entry $table $context $table_key $key_columns $key_value
			return $rowCount
		}

		# Update in cache only
		if {!$dirtyFlag} {
			# Nothing updated
			set affected_rows 0
		} else {
			# affected_rows is matching rows
			set affected_rows $rowCount
		}
		set cache_entry [list $dirtyFlag $dirtyList $intoTime $allColumns $order $rowCount $newRowData]
		set ${table_key}($key_value) $cache_entry
	} else {
		# Perform update to database, affected_rows is rows actually changed
		set affected_rows [perform_update $table merged $context $key_columns $key_value $update_columns]
	}

	# Throw error if affected_rows < min or > max (the update is still performed)
	if {$affected_rows < $min_rows} {
		error "update: updated $affected_rows row(s) minimum $min_rows"
	}
	if {$max_rows >= 0 && $affected_rows > $max_rows} {
		error "update: updated $affected_rows row(s) maximum $max_rows"
	}
	# May be number of matching rows or rows actually changed
	return $affected_rows
}

# Delete zero or more rows, in cache or database
# Build sql query string
# Return number of rows affected
# Arguments and options:
#   row            Name of array with key columns
#   table          Database table name
#   key            List of column names forming the query key (the where clause)
#   key_data       Name of array with key columns (name and value)
#   <column>       Key-column name and value may be supplied as an option
#   min            Error if deleted less than min rows
#   max            Error if deleted greater than max rows (-1 = no maximum)
#   nocache        If true don't use cache
proc delete { args } {
	# Get arguments and options
	set defaults {table "" key_data "" min 0 max -1 nocache 0}
	command::arguments $args opt {row table} $defaults
	set connID [command::inherit property connID]
	global msecNow

	# Row data may supply key columns
	upvar $opt(row) deleterow

	# Table name defaults to name of row array
	set table $opt(table)
	if {$table == ""} {
		set table [string toupper $opt(row)]
	}

	# Use default key from schema
	if {[info exists opt(key)]} {
		set key_columns $opt(key)
	} else {
		set key_columns $::schema::key_columns($table)
	}
	set key_columns [schema::get_sorted_key $table $key_columns]

	# Key column names and values, default to row array
	if {$opt(key_data) == ""} {
		upvar $opt(row) key_data
	} else {
		upvar $opt(key_data) key_data
	}

	set min_rows $opt(min)
	set max_rows $opt(max)

	# Bypass cache
	set nocache $opt(nocache)

	# Determine query key
	set key_value [make_query_key $table $key_columns opt key_data]

	# -------------------------------------------------------
	# Check cache

	# if nocache just do the query ...
	
	set cache_ns [make_cache_ns $connID]
	set table_ns [make_table_ns $cache_ns $table]
	set table_key [get_table_key $table_ns $key_columns]
	set context [get_calling_context "::query::delete"]

	# Flush and clear matching other-keys for this table
	foreach other_columns [all_table_keys $table_ns] {
		if {$other_columns == $key_columns} { continue }
		set pattern [make_key_pattern $key_columns $other_columns $key_value]
		set matchList [get_matching_entries $table_ns $other_columns $pattern]
				
		set other_key [get_table_key $table_ns $other_columns]			
		foreach name $matchList {
			clear_cache_entry $table $context $other_key $other_columns $name
		}
	}
	
	# Find existing cache entry
	set found 0
	set dirtyFlag 0
	if {[info exists ${table_key}($key_value)]} {
		set cache_entry [set ${table_key}($key_value)]
		set intoTime [lindex $cache_entry 2]
		set rowCount [lindex $cache_entry 5]

		set msecNow [clock clicks -milliseconds]
		if {[alive $cache_ns $table $intoTime]} {
			set found 1
		}
	}

	# -------------------------------------------------------
	# Write to cache or database

	if {$found} {
		set affected_rows $rowCount
		# Leave cache un-touched, return zero rows affected
		if {$affected_rows == 0} {
			if {$affected_rows < $min_rows} {
				error "delete: deleted $affected_rows row(s) minimum $min_rows"
			}
			return $affected_rows
		}
		# Update cache
		set dirtyFlag 1
	} else {
		# Perform delete to database
		set affected_rows [perform_delete $table $context $key_columns $key_value]
		set dirtyFlag 0
		set msecNow [clock clicks -milliseconds]
		set intoTime $msecNow
	}

	# Write to cache
	set dirtyList {}
	set allColumns 0
	set rowCount 0
	set rowData {}
	set order {}
	set cache_entry [list $dirtyFlag $dirtyList $intoTime $allColumns $order $rowCount $rowData]
	set ${table_key}($key_value) $cache_entry

	# Throw error if affected_rows < min or > max (the delete is still performed)
	if {$affected_rows < $min_rows} {
		error "delete: deleted $affected_rows row(s) minimum $min_rows"
	}
	if {$max_rows >= 0 && $affected_rows > $max_rows} {
		error "delete: deleted $affected_rows row(s) maximum $max_rows"
	}
	return $affected_rows
}

# Set default column values for row array
# Arguments and options:
#   row            Name of array where default column values returned
#   table          Database table name
proc null { args } {
	# Get arguments and options
	set defaults {table ""}
	command::arguments $args opt {row table} $defaults

	# Where default row is returned
	upvar $opt(row) row

	# Table name defaults to name of row array
	set table $opt(table)
	if {$table == ""} {
		set table [string toupper $opt(row)]
	}

	# Default value is per schema or per column type
	foreach column $::schema::all_columns($table) {
		set row($column) [set ::schema::default_${table}($column)]
	}
}

# -------------------------------------------------------------------------
# Internal procs - building keys

# Create & return list of key values
proc make_query_key { table key_columns optName keyDataName } {
	upvar $optName opt
	upvar $keyDataName keyData

	set key_value {}
	foreach name $key_columns {
		if {[info exists opt($name)]} {
			set value $opt($name)
		} else {
			if {[catch {set value $keyData($name) }]} { error "key column $name missing" }
		}
		# Map value per type
		lappend key_value [canonicalise $table $name $value]
	}
	return $key_value
}

# Canonicalise the value, according to its type
# Scan %d and %f returns {} if invalid number, produces sql error ...
# No canonicalising for date/datetime/timestamp
proc canonicalise { table column value } {
	switch -- [set ::schema::type_${table}($column)] {
		0 { return [string trim $value] }
		1 { return [scan $value %d] }
		2 { return [scan $value %f] }
		3 { return [string trim $value] }
	}
}

# Create cache namespace, per database connection
proc make_cache_ns { connID } {
	set cache_ns "::cache.${connID}"
	namespace eval $cache_ns {}
	return $cache_ns
}

# Create table namespace, within cache namespace
proc make_table_ns { cache_ns table } {
	set table_ns "${cache_ns}::${table}"
	namespace eval $table_ns {}
	return $table_ns
}

# Return name of table_key array; key_columns is array name
proc get_table_key { table_ns key_columns } {
	set table_key "${table_ns}::${key_columns}"
	return $table_key
}

# Get list of all tables in the cache
proc all_tables { cache_ns } {
	set tableList {}
	foreach table_ns [namespace children $cache_ns] {
		lappend tableList [namespace tail $table_ns]
	}
	return $tableList
}

# Get list of all key_columns for the table
proc all_table_keys { table_ns } {
	set all_keys {}

	# Find all variables in the table namespace
	set varList [info vars "${table_ns}::*"]
	foreach var $varList {
		if {[array exists $var]} {
			set key_columns [namespace tail $var]
			lappend all_keys $key_columns
		}
	}
	return $all_keys
}

# Build pattern for matching two keys
proc make_key_pattern { key_columns other_columns key_value } {
	set pattern {}
	foreach this_name $key_columns other_name $other_columns value $key_value {
		if {$this_name == $other_name} {
			lappend pattern $value
		} else {
			# Not all other-keys were matched
			if {$other_name != "" && $pattern != ""} { lappend pattern "*" }
			break
		}
	}
	return $pattern
}

# Return list of all key entries matching the pattern
proc get_matching_entries { table_ns other_columns pattern } {
	set var "${table_ns}::${other_columns}"
	# If no columns in common, match all entries
	if {$pattern == ""} {
		set matchList [array names $var]
	} else {
		set matchList [array names $var $pattern]
	}
	return $matchList
}

# -------------------------------------------------------------------------

# Flush or clear cache entries
# Default action is flush (if neither clear nor expire is true)
# Arguments and options:
#   table          Database table name (default all)
#   clear          Clear all entries (as well as flush)
#   expire         Flush and clear expired entries
#   suppress       Suppress database error messages

proc flush_cache { args } {
	# Get arguments and options
	set defaults {table "" clear 0 expire 0 suppress 0}
	command::arguments $args opt {table} $defaults
	set connID [command::inherit property connID]
	global msecNow
	
	set table $opt(table)
	set clear $opt(clear)
	set expire $opt(expire)
	set suppress $opt(suppress)

	set cache_ns [make_cache_ns $connID]
	set context [get_calling_context "::query::flush_cache"]
	set msecNow [clock clicks -milliseconds]

	if {$table == ""} {
		foreach table [all_tables $cache_ns] {
			flush_table $cache_ns $table $clear $expire $suppress $context
		}
	} else {
		flush_table $cache_ns $table $clear $expire $suppress $context
	}
}

# Flush or clear cache entries for single table
# Pass suppress option ...
proc flush_table { cache_ns table clear expire suppress context } {
	set table_ns [make_table_ns $cache_ns $table]
		
	# All keys for this table
	foreach key_columns [all_table_keys $table_ns] {
		set table_key [get_table_key $table_ns $key_columns]			
		foreach key_value [array names $table_key] {
			if {$clear} {
				clear_cache_entry $table $context $table_key $key_columns $key_value
			} elseif {$expire} {
				# Has the cache entry expired ?
				set cache_entry [set ${table_key}($key_value)]
				set intoTime [lindex $cache_entry 2]
				if {![alive $cache_ns $table $intoTime]} {
					clear_cache_entry $table $context $table_key $key_columns $key_value
				}
			} else {
				flush_cache_entry $table $context $table_key $key_columns $key_value
			}
		}
	}
}

# Return true if the entry is alive, ie. hasn't expired
# Expiry is in seconds, resolution is msecs
# Assume msecNow has been set recently
proc alive { cache_ns table intoTime } {
	global msecNow

	set expiry 60
	# Expiry may be set on the cache instance or the table
	if {[info exists ${cache_ns}::expiry]} {
		set expiry [set ${cache_ns}::expiry]
	} elseif {[info exists ::schema::expiry($table)]} {
		set expiry $::schema::expiry($table)
	}
	set duration [expr {($msecNow - $intoTime) / 1000.0}]
	return [expr {$duration >= 0 && $duration < $expiry}]
}

# Flush and clear the cache entry
proc clear_cache_entry { table context table_key key_columns key_value } {
	if {![info exists ${table_key}($key_value)]} { return }

	set cache_entry [set ${table_key}($key_value)]
	set dirtyFlag [lindex $cache_entry 0]
	if {$dirtyFlag} {
		flush_cache_entry $table $context $table_key $key_columns $key_value
	}
	# Clear entry
	unset -nocomplain -- ${table_key}($key_value)
}

# Flush cache entry, leave it clean
# Perform insert, update or delete as required
proc flush_cache_entry { table context table_key key_columns key_value } {
	if {![info exists ${table_key}($key_value)]} { return }
	global msecNow

	# Nothing to do if cache entry is already clean
	set cache_entry [set ${table_key}($key_value)]
	set dirtyFlag [lindex $cache_entry 0]
	if {!$dirtyFlag} { return }

	set dirtyList [lindex $cache_entry 1]
	set intoTime [lindex $cache_entry 2]
	set allColumns [lindex $cache_entry 3]
	set orderBy [lindex $cache_entry 4]
	set rowCount [lindex $cache_entry 5]
	set rowData [lindex $cache_entry 6]

	if {$rowCount == 0} {
		# Delete
		set affected_rows [perform_delete $table $context $key_columns $key_value]
	} elseif {$dirtyList != ""} {
		# Update
		array set first_row [lindex $rowData 0]
		set update_columns $dirtyList
		set affected_rows [perform_update $table first_row $context $key_columns $key_value $update_columns]
		# affected_rows may be less (rows actually changed vs matching)
		if {$affected_rows > $rowCount} {
			error "flush_cache_entry: update affected $affected_rows rows expecting $rowCount"
		} 
	} else {
		# Insert
		if {$rowCount > 1} { error "flush_cache_entry: expecting 1 row to insert got $rowCount" }
		array set row [lindex $rowData 0]
		set insert_id [perform_insert $table row $context]
	}

	# Set cache entry clean
	set dirtyFlag 0
	set dirtyList {}
	set msecNow [clock clicks -milliseconds]
	set intoTime $msecNow
	set cache_entry [list $dirtyFlag $dirtyList $intoTime $allColumns $orderBy $rowCount $rowData]
	set ${table_key}($key_value) $cache_entry
}

# -------------------------------------------------------------------------
# Save and restore cache

# Save current contents of cache to file
# First flush any dirty entries; expire old entries
# Caller should include database and session ID in fileName
proc save_cache { fileName } {
	set connID [command::inherit property connID]
	global msecNow
	
	set cache_ns [make_cache_ns $connID]
	set context [get_calling_context "::query::save_cache"]
	set msecNow [clock clicks -milliseconds]
	set table_list {}
	
	foreach table [all_tables $cache_ns] {
		# All key_columns for this table
		set table_ns [make_table_ns $cache_ns $table]
		foreach key_columns [all_table_keys $table_ns] {
			set table_key [get_table_key $table_ns $key_columns]
			set table_key_list {}

			# Each cache entry
			foreach key_value [array names $table_key] {
				flush_cache_entry $table $context $table_key $key_columns $key_value

				# Expire cache entry
				set cache_entry [set ${table_key}($key_value)]
				set intoTime [lindex $cache_entry 2]
				if {![alive $cache_ns $table $intoTime]} {
					unset -nocomplain -- ${table_key}($key_value)
					continue
				}
				lappend table_key_list $key_value
				lappend table_key_list $cache_entry
			}
			lappend table_list [list $table $key_columns $table_key_list]
		}
	}

	# Write contents of cache to file
	set fid [open $fileName w]
	fconfigure $fid -translation binary
	puts -nonewline $fid $table_list
	close $fid
}

# Restore contents of cache from file
# Check saved cache entries are valid
proc load_cache { fileName } {
	set connID [command::inherit property connID]
	set cache_ns [make_cache_ns $connID]

	if {![file exists $fileName]} { return }
	# Read contents of file
	set fid [open $fileName]
	fconfigure $fid -translation binary
	set table_list [read $fid]
	close $fid

	# Each table
	foreach table_item $table_list {
		set table [lindex $table_item 0]
		set key_columns [lindex $table_item 1]
		set table_key_list [lindex $table_item 2]

		# Create table namespace and table key array
		set table_ns [make_table_ns $cache_ns $table]
		set table_key [get_table_key $table_ns $key_columns]

		# Restore cache entries (if valid)
		foreach {name value} $table_key_list {
			set length [llength $value]
			set rowCount [lindex $value 5]
			if {$length == 7 && $rowCount >= 0 && $rowCount < 5000} {
				set ${table_key}($name) $value
			}
		}
	}
}

# -------------------------------------------------------------------------
# Construct and perform database queries

# Build and perform insert query, return insert_id
proc perform_insert { table rowName context } {
	upvar $rowName row
	command::inherit property {connID}

	# Build list of insert columns
	set columnList [lsort [array names row]]
	set valueList {}
	# Quote string values etc
	foreach name $columnList {
		set value $row($name)
		set type [set ::schema::type_${table}($name)]
		if {$value == ""} {
			# Set numbers to zero, dates to null
			switch -- $type {
				1 { lappend valueList 0 }
				2 { lappend valueList 0.0 }
				3 { lappend valueList "null" }
				default { lappend valueList "''" }
			}
		} else {
			# Quote strings and dates
			if {$type == 1 || $type == 2} {
				lappend valueList $value
			} else {
				lappend valueList "'[sql_escape $value]'"
			}
		}
	}
	set insert_columns [join $columnList ","]
	set insert_values [join $valueList ","]

	# Build insert query
	set statement "insert into ${table} (${insert_columns}) values (${insert_values})"

	# Perform query either via fbserver or fbproxy
	# May fail with duplicate error
	if {[setup remote_host] != ""} {
		set result [fbremote::sql_list $statement]
	} else {
		set result [sql_list_query $property(connID) $statement $context]
	}
	set insert_id [query::insert_id $result]
	return $insert_id
}

# Build and perform update query, return affected_rows
proc perform_update { table rowName context key_columns key_value update_columns } {
	upvar $rowName row
	command::inherit property {connID}

	# Quote string values etc
	set setList {}
	foreach name [lsort $update_columns] {
		set value $row($name)
		set type [set ::schema::type_${table}($name)]
		if {$value == ""} {
			# Set numbers to zero, dates to null
			switch -- $type {
				1 { lappend setList "$name = 0" }
				2 { lappend setList "$name = 0.0" }
				3 { lappend setList "$name = null" }
				default { lappend setList "$name = ''" }
			}
		} else {
			# Quote strings and dates
			if {$type == 1 || $type == 2} {
				lappend setList "$name = $value"
			} else {
				lappend setList "$name = '[sql_escape $value]'"
			}
		}
	}
	set set_clause [join $setList ","]

	# Build update query
	set where [make_where_clause $table $key_columns $key_value]
	set statement "update ${table} set ${set_clause}${where}"

	if {$where == "none"} {
		# Query would affect zero rows
		set result [list 0 0]
	} else {
		# Perform query either via fbserver or fbproxy
		if {[setup remote_host] != ""} {
			set result [fbremote::sql_list $statement]
		} else {
			set result [sql_list_query $property(connID) $statement $context]
		}
	}

	set affected_rows [query::affected_rows $result]
	return $affected_rows
}

# Build and perform delete query, return affected_rows
proc perform_delete { table context key_columns key_value } {
	command::inherit property {connID}

	set where [make_where_clause $table $key_columns $key_value]
	set statement "delete from ${table}${where}"

	if {$where == "none"} {
		# Query would affect zero rows
		set result [list 0 0]
	} else {
		# Perform query either via fbserver or fbproxy
		# May fail with duplicate error
		if {[setup remote_host] != ""} {
			set result [fbremote::sql_list $statement]
		} else {
			set result [sql_list_query $property(connID) $statement $context]
		}
	}
	set affected_rows [query::affected_rows $result]
	return $affected_rows
}

# Build where clause, map each key_column to "column = value"
# Return "none" if the query would match zero rows
proc make_where_clause { table key_columns key_value } {
	if {$key_columns == ""} { return "" }
	set clauseList {}
	foreach name $key_columns value $key_value {
		# Quote the value, test for null
		set null [set ::schema::null_${table}($name)]
		switch -- [set ::schema::type_${table}($name)] {
			0 {
				# String, null means empty string
				if {$value == "" && $null} {
					lappend clauseList "($name = '' or $name is null)"
				} else {
					lappend clauseList "$name = '[sql_escape $value]'"
				}
			}
			1 {
				# Integer, null in the database means zero
				if {$value == ""} {
					return "none"
				} elseif {$value == 0 && $null} {
					lappend clauseList "($name = 0 or $name is null)"
				} else {
					lappend clauseList "$name = $value"
				}
			}
			2 {
				# Real, null in the database means zero
				if {$value == ""} {
					return "none"
				} elseif {$value == 0.0 && $null} {
					lappend clauseList "($name = 0.0 or $name is null)"
				} else {
					lappend clauseList "$name = $value"
				}
			}
			3 {
				# Date, null means empty string
				# If column is not-null, cannot match empty string
				if {$value == "" && $null} {
					lappend clauseList "$name is null"
				} else {
					lappend clauseList "$name = '[sql_escape $value]'"
				}
			}
		}
	}
	set clause " where [join $clauseList { and }]"
	return $clause
}

# Build order by clause, each column may be ascending or descending
proc make_order_clause { orderList } {
	if {$orderList == ""} { return "" }
	set clauseList {}
	foreach name $orderList {
		# ^column = descending
		if {[string match {^[A-Za-z]*} $name]} {
			lappend clauseList "[string range $name 1 end] desc"
		} else {
			lappend clauseList $name
		}
	}
	set clause " order by [join $clauseList {,}]"
	return $clause
}

# Replace any single-quote with two single-quotes
# Backslash - replace 1-2 or 2-4 ? ...
proc sql_escape { value } {
	regsub -all {'} $value "''" value
	regsub -all "\\\\" $value {\\\\} value
	return $value
}

# End namespace
}

# -------------------------------------------------------------------------

